var __defProp = Object.defineProperty;
var __getOwnPropDesc = Object.getOwnPropertyDescriptor;
var __decorateClass = (decorators, target, key, kind) => {
  var result = kind > 1 ? void 0 : kind ? __getOwnPropDesc(target, key) : target;
  for (var i = decorators.length - 1, decorator; i >= 0; i--)
    if (decorator = decorators[i])
      result = (kind ? decorator(target, key, result) : decorator(result)) || result;
  if (kind && result) __defProp(target, key, result);
  return result;
};
import { EMPTY_ARRAY, atom, computed, react, transact, unsafe__withoutCapture } from "@tldraw/state";
import {
  reverseRecordsDiff
} from "@tldraw/store";
import {
  CameraRecordType,
  InstancePageStateRecordType,
  PageRecordType,
  TLDOCUMENT_ID,
  TLINSTANCE_ID,
  TLPOINTER_ID,
  createBindingId,
  createShapeId,
  getShapePropKeysByStyle,
  isPageId,
  isShapeId
} from "@tldraw/tlschema";
import {
  FileHelpers,
  PerformanceTracker,
  Result,
  Timers,
  annotateError,
  assert,
  assertExists,
  compact,
  dedupe,
  exhaustiveSwitchError,
  fetch,
  getIndexAbove,
  getIndexBetween,
  getIndices,
  getIndicesAbove,
  getIndicesBetween,
  getOwnProperty,
  hasOwnProperty,
  last,
  lerp,
  sortById,
  sortByIndex,
  structuredClone
} from "@tldraw/utils";
import EventEmitter from "eventemitter3";
import { flushSync } from "react-dom";
import { createRoot } from "react-dom/client";
import { getSnapshot, loadSnapshot } from "../config/TLEditorSnapshot.mjs";
import { createTLUser } from "../config/createTLUser.mjs";
import { checkBindings } from "../config/defaultBindings.mjs";
import { checkShapesAndAddCore } from "../config/defaultShapes.mjs";
import {
  DEFAULT_ANIMATION_OPTIONS,
  DEFAULT_CAMERA_OPTIONS,
  INTERNAL_POINTER_IDS,
  LEFT_MOUSE_BUTTON,
  MIDDLE_MOUSE_BUTTON,
  RIGHT_MOUSE_BUTTON,
  STYLUS_ERASER_BUTTON,
  ZOOM_TO_FIT_PADDING
} from "../constants.mjs";
import { defaultTldrawOptions } from "../options.mjs";
import { Box } from "../primitives/Box.mjs";
import { Mat } from "../primitives/Mat.mjs";
import { Vec } from "../primitives/Vec.mjs";
import { EASINGS } from "../primitives/easings.mjs";
import { Group2d } from "../primitives/geometry/Group2d.mjs";
import { intersectPolygonPolygon } from "../primitives/intersect.mjs";
import { PI2, approximately, areAnglesCompatible, clamp, pointInPolygon } from "../primitives/utils.mjs";
import { SharedStyleMap } from "../utils/SharedStylesMap.mjs";
import { dataUrlToFile } from "../utils/assets.mjs";
import { debugFlags } from "../utils/debug-flags.mjs";
import { getIncrementedName } from "../utils/getIncrementedName.mjs";
import { getReorderingShapesChanges } from "../utils/reorderShapes.mjs";
import { applyRotationToSnapshotShapes, getRotationSnapshot } from "../utils/rotation.mjs";
import { uniqueId } from "../utils/uniqueId.mjs";
import { bindingsIndex } from "./derivations/bindingsIndex.mjs";
import { notVisibleShapes } from "./derivations/notVisibleShapes.mjs";
import { parentsToChildren } from "./derivations/parentsToChildren.mjs";
import { deriveShapeIdsInCurrentPage } from "./derivations/shapeIdsInCurrentPage.mjs";
import { getSvgJsx } from "./getSvgJsx.mjs";
import { ClickManager } from "./managers/ClickManager.mjs";
import { EdgeScrollManager } from "./managers/EdgeScrollManager.mjs";
import { EnvironmentManager } from "./managers/EnvironmentManager.mjs";
import { FocusManager } from "./managers/FocusManager.mjs";
import { HistoryManager } from "./managers/HistoryManager.mjs";
import { ScribbleManager } from "./managers/ScribbleManager.mjs";
import { SnapManager } from "./managers/SnapManager/SnapManager.mjs";
import { TextManager } from "./managers/TextManager.mjs";
import { TickManager } from "./managers/TickManager.mjs";
import { UserPreferencesManager } from "./managers/UserPreferencesManager.mjs";
import { RootState } from "./tools/RootState.mjs";
class Editor extends EventEmitter {
  constructor({
    store,
    user,
    shapeUtils,
    bindingUtils,
    tools,
    getContainer,
    cameraOptions,
    initialState,
    autoFocus,
    inferDarkMode,
    options
  }) {
    super();
    this.options = { ...defaultTldrawOptions, ...options };
    this.store = store;
    this.disposables.add(this.store.dispose.bind(this.store));
    this.history = new HistoryManager({
      store,
      annotateError: (error) => {
        this.annotateError(error, { origin: "history.batch", willCrashApp: true });
        this.crash(error);
      }
    });
    this.snaps = new SnapManager(this);
    this.timers = new Timers();
    this.disposables.add(this.timers.dispose.bind(this.timers));
    this._cameraOptions.set({ ...DEFAULT_CAMERA_OPTIONS, ...cameraOptions });
    this.user = new UserPreferencesManager(user ?? createTLUser(), inferDarkMode ?? false);
    this.getContainer = getContainer ?? (() => document.body);
    this.textMeasure = new TextManager(this);
    this._tickManager = new TickManager(this);
    class NewRoot extends RootState {
      static initial = initialState ?? "";
    }
    this.root = new NewRoot(this);
    this.root.children = {};
    const allShapeUtils = checkShapesAndAddCore(shapeUtils);
    const _shapeUtils = {};
    const _styleProps = {};
    const allStylesById = /* @__PURE__ */ new Map();
    for (const Util of allShapeUtils) {
      const util = new Util(this);
      _shapeUtils[Util.type] = util;
      const propKeysByStyle = getShapePropKeysByStyle(Util.props ?? {});
      _styleProps[Util.type] = propKeysByStyle;
      for (const style of propKeysByStyle.keys()) {
        if (!allStylesById.has(style.id)) {
          allStylesById.set(style.id, style);
        } else if (allStylesById.get(style.id) !== style) {
          throw Error(
            `Multiple style props with id "${style.id}" in use. Style prop IDs must be unique.`
          );
        }
      }
    }
    this.shapeUtils = _shapeUtils;
    this.styleProps = _styleProps;
    const allBindingUtils = checkBindings(bindingUtils);
    const _bindingUtils = {};
    for (const Util of allBindingUtils) {
      const util = new Util(this);
      _bindingUtils[Util.type] = util;
    }
    this.bindingUtils = _bindingUtils;
    for (const Tool of [...tools]) {
      if (hasOwnProperty(this.root.children, Tool.id)) {
        throw Error(`Can't override tool with id "${Tool.id}"`);
      }
      this.root.children[Tool.id] = new Tool(this, this.root);
    }
    this.environment = new EnvironmentManager(this);
    this.scribbles = new ScribbleManager(this);
    const cleanupInstancePageState = (prevPageState, shapesNoLongerInPage) => {
      let nextPageState = null;
      const selectedShapeIds = prevPageState.selectedShapeIds.filter(
        (id) => !shapesNoLongerInPage.has(id)
      );
      if (selectedShapeIds.length !== prevPageState.selectedShapeIds.length) {
        if (!nextPageState) nextPageState = { ...prevPageState };
        nextPageState.selectedShapeIds = selectedShapeIds;
      }
      const erasingShapeIds = prevPageState.erasingShapeIds.filter(
        (id) => !shapesNoLongerInPage.has(id)
      );
      if (erasingShapeIds.length !== prevPageState.erasingShapeIds.length) {
        if (!nextPageState) nextPageState = { ...prevPageState };
        nextPageState.erasingShapeIds = erasingShapeIds;
      }
      if (prevPageState.hoveredShapeId && shapesNoLongerInPage.has(prevPageState.hoveredShapeId)) {
        if (!nextPageState) nextPageState = { ...prevPageState };
        nextPageState.hoveredShapeId = null;
      }
      if (prevPageState.editingShapeId && shapesNoLongerInPage.has(prevPageState.editingShapeId)) {
        if (!nextPageState) nextPageState = { ...prevPageState };
        nextPageState.editingShapeId = null;
      }
      const hintingShapeIds = prevPageState.hintingShapeIds.filter(
        (id) => !shapesNoLongerInPage.has(id)
      );
      if (hintingShapeIds.length !== prevPageState.hintingShapeIds.length) {
        if (!nextPageState) nextPageState = { ...prevPageState };
        nextPageState.hintingShapeIds = hintingShapeIds;
      }
      if (prevPageState.focusedGroupId && shapesNoLongerInPage.has(prevPageState.focusedGroupId)) {
        if (!nextPageState) nextPageState = { ...prevPageState };
        nextPageState.focusedGroupId = null;
      }
      return nextPageState;
    };
    this.sideEffects = this.store.sideEffects;
    let deletedBindings = /* @__PURE__ */ new Map();
    const deletedShapeIds = /* @__PURE__ */ new Set();
    const invalidParents = /* @__PURE__ */ new Set();
    let invalidBindingTypes = /* @__PURE__ */ new Set();
    this.disposables.add(
      this.sideEffects.registerOperationCompleteHandler(() => {
        deletedShapeIds.clear();
        for (const parentId of invalidParents) {
          invalidParents.delete(parentId);
          const parent = this.getShape(parentId);
          if (!parent) continue;
          const util = this.getShapeUtil(parent);
          const changes = util.onChildrenChange?.(parent);
          if (changes?.length) {
            this.updateShapes(changes);
          }
        }
        if (invalidBindingTypes.size) {
          const t = invalidBindingTypes;
          invalidBindingTypes = /* @__PURE__ */ new Set();
          for (const type of t) {
            const util = this.getBindingUtil(type);
            util.onOperationComplete?.();
          }
        }
        if (deletedBindings.size) {
          const t = deletedBindings;
          deletedBindings = /* @__PURE__ */ new Map();
          for (const opts of t.values()) {
            this.getBindingUtil(opts.binding).onAfterDelete?.(opts);
          }
        }
        this.emit("update");
      })
    );
    this.disposables.add(
      this.sideEffects.register({
        shape: {
          afterChange: (shapeBefore, shapeAfter) => {
            for (const binding of this.getBindingsInvolvingShape(shapeAfter)) {
              invalidBindingTypes.add(binding.type);
              if (binding.fromId === shapeAfter.id) {
                this.getBindingUtil(binding).onAfterChangeFromShape?.({
                  binding,
                  shapeBefore,
                  shapeAfter
                });
              }
              if (binding.toId === shapeAfter.id) {
                this.getBindingUtil(binding).onAfterChangeToShape?.({
                  binding,
                  shapeBefore,
                  shapeAfter
                });
              }
            }
            if (shapeBefore.parentId !== shapeAfter.parentId) {
              const notifyBindingAncestryChange = (id) => {
                const descendantShape = this.getShape(id);
                if (!descendantShape) return;
                for (const binding of this.getBindingsInvolvingShape(descendantShape)) {
                  invalidBindingTypes.add(binding.type);
                  if (binding.fromId === descendantShape.id) {
                    this.getBindingUtil(binding).onAfterChangeFromShape?.({
                      binding,
                      shapeBefore: descendantShape,
                      shapeAfter: descendantShape
                    });
                  }
                  if (binding.toId === descendantShape.id) {
                    this.getBindingUtil(binding).onAfterChangeToShape?.({
                      binding,
                      shapeBefore: descendantShape,
                      shapeAfter: descendantShape
                    });
                  }
                }
              };
              notifyBindingAncestryChange(shapeAfter.id);
              this.visitDescendants(shapeAfter.id, notifyBindingAncestryChange);
            }
            if (shapeBefore.parentId !== shapeAfter.parentId && isPageId(shapeAfter.parentId)) {
              const allMovingIds = /* @__PURE__ */ new Set([shapeBefore.id]);
              this.visitDescendants(shapeBefore.id, (id) => {
                allMovingIds.add(id);
              });
              for (const instancePageState of this.getPageStates()) {
                if (instancePageState.pageId === shapeAfter.parentId) continue;
                const nextPageState = cleanupInstancePageState(instancePageState, allMovingIds);
                if (nextPageState) {
                  this.store.put([nextPageState]);
                }
              }
            }
            if (shapeBefore.parentId && isShapeId(shapeBefore.parentId)) {
              invalidParents.add(shapeBefore.parentId);
            }
            if (shapeAfter.parentId !== shapeBefore.parentId && isShapeId(shapeAfter.parentId)) {
              invalidParents.add(shapeAfter.parentId);
            }
          },
          beforeDelete: (shape) => {
            if (deletedShapeIds.has(shape.id)) return;
            if (shape.parentId && isShapeId(shape.parentId)) {
              invalidParents.add(shape.parentId);
            }
            deletedShapeIds.add(shape.id);
            const deleteBindingIds = [];
            for (const binding of this.getBindingsInvolvingShape(shape)) {
              invalidBindingTypes.add(binding.type);
              deleteBindingIds.push(binding.id);
              const util = this.getBindingUtil(binding);
              if (binding.fromId === shape.id) {
                util.onBeforeIsolateToShape?.({ binding, removedShape: shape });
                util.onBeforeDeleteFromShape?.({ binding, shape });
              } else {
                util.onBeforeIsolateFromShape?.({ binding, removedShape: shape });
                util.onBeforeDeleteToShape?.({ binding, shape });
              }
            }
            if (deleteBindingIds.length) {
              this.deleteBindings(deleteBindingIds);
            }
            const deletedIds = /* @__PURE__ */ new Set([shape.id]);
            const updates = compact(
              this.getPageStates().map((pageState) => {
                return cleanupInstancePageState(pageState, deletedIds);
              })
            );
            if (updates.length) {
              this.store.put(updates);
            }
          }
        },
        binding: {
          beforeCreate: (binding) => {
            const next = this.getBindingUtil(binding).onBeforeCreate?.({ binding });
            if (next) return next;
            return binding;
          },
          afterCreate: (binding) => {
            invalidBindingTypes.add(binding.type);
            this.getBindingUtil(binding).onAfterCreate?.({ binding });
          },
          beforeChange: (bindingBefore, bindingAfter) => {
            const updated = this.getBindingUtil(bindingAfter).onBeforeChange?.({
              bindingBefore,
              bindingAfter
            });
            if (updated) return updated;
            return bindingAfter;
          },
          afterChange: (bindingBefore, bindingAfter) => {
            invalidBindingTypes.add(bindingAfter.type);
            this.getBindingUtil(bindingAfter).onAfterChange?.({ bindingBefore, bindingAfter });
          },
          beforeDelete: (binding) => {
            this.getBindingUtil(binding).onBeforeDelete?.({ binding });
          },
          afterDelete: (binding) => {
            this.getBindingUtil(binding).onAfterDelete?.({ binding });
            invalidBindingTypes.add(binding.type);
          }
        },
        page: {
          afterCreate: (record) => {
            const cameraId = CameraRecordType.createId(record.id);
            const _pageStateId = InstancePageStateRecordType.createId(record.id);
            if (!this.store.has(cameraId)) {
              this.store.put([CameraRecordType.create({ id: cameraId })]);
            }
            if (!this.store.has(_pageStateId)) {
              this.store.put([
                InstancePageStateRecordType.create({ id: _pageStateId, pageId: record.id })
              ]);
            }
          },
          afterDelete: (record, source) => {
            if (this.getInstanceState()?.currentPageId === record.id) {
              const backupPageId = this.getPages().find((p) => p.id !== record.id)?.id;
              if (backupPageId) {
                this.store.put([{ ...this.getInstanceState(), currentPageId: backupPageId }]);
              } else if (source === "user") {
                this.store.ensureStoreIsUsable();
              }
            }
            const cameraId = CameraRecordType.createId(record.id);
            const instance_PageStateId = InstancePageStateRecordType.createId(record.id);
            this.store.remove([cameraId, instance_PageStateId]);
          }
        },
        instance: {
          afterChange: (prev, next, source) => {
            if (!this.store.has(next.currentPageId)) {
              const backupPageId = this.store.has(prev.currentPageId) ? prev.currentPageId : this.getPages()[0]?.id;
              if (backupPageId) {
                this.store.update(next.id, (instance) => ({
                  ...instance,
                  currentPageId: backupPageId
                }));
              } else if (source === "user") {
                this.store.ensureStoreIsUsable();
              }
            }
          }
        },
        instance_page_state: {
          afterChange: (prev, next) => {
            if (prev?.selectedShapeIds !== next?.selectedShapeIds) {
              const filtered = next.selectedShapeIds.filter((id) => {
                let parentId = this.getShape(id)?.parentId;
                while (isShapeId(parentId)) {
                  if (next.selectedShapeIds.includes(parentId)) {
                    return false;
                  }
                  parentId = this.getShape(parentId)?.parentId;
                }
                return true;
              });
              let nextFocusedGroupId = null;
              if (filtered.length > 0) {
                const commonGroupAncestor = this.findCommonAncestor(
                  compact(filtered.map((id) => this.getShape(id))),
                  (shape) => this.isShapeOfType(shape, "group")
                );
                if (commonGroupAncestor) {
                  nextFocusedGroupId = commonGroupAncestor;
                }
              } else {
                if (next?.focusedGroupId) {
                  nextFocusedGroupId = next.focusedGroupId;
                }
              }
              if (filtered.length !== next.selectedShapeIds.length || nextFocusedGroupId !== next.focusedGroupId) {
                this.store.put([
                  {
                    ...next,
                    selectedShapeIds: filtered,
                    focusedGroupId: nextFocusedGroupId ?? null
                  }
                ]);
              }
            }
          }
        }
      })
    );
    this._currentPageShapeIds = deriveShapeIdsInCurrentPage(
      this.store,
      () => this.getCurrentPageId()
    );
    this._parentIdsToChildIds = parentsToChildren(this.store);
    this.disposables.add(
      this.store.listen((changes) => {
        this.emit("change", changes);
      })
    );
    this.disposables.add(this.history.dispose);
    this.run(
      () => {
        this.store.ensureStoreIsUsable();
        this._updateCurrentPageState({
          editingShapeId: null,
          hoveredShapeId: null,
          erasingShapeIds: []
        });
      },
      { history: "ignore" }
    );
    if (initialState && this.root.children[initialState] === void 0) {
      throw Error(`No state found for initialState "${initialState}".`);
    }
    this.root.enter(void 0, "initial");
    this.edgeScrollManager = new EdgeScrollManager(this);
    this.focusManager = new FocusManager(this, autoFocus);
    this.disposables.add(this.focusManager.dispose.bind(this.focusManager));
    if (this.getInstanceState().followingUserId) {
      this.stopFollowingUser();
    }
    this.on("tick", this._flushEventsForTick);
    this.timers.requestAnimationFrame(() => {
      this._tickManager.start();
    });
    this.performanceTracker = new PerformanceTracker();
  }
  options;
  /**
   * The editor's store
   *
   * @public
   */
  store;
  /**
   * The root state of the statechart.
   *
   * @public
   */
  root;
  /**
   * A set of functions to call when the app is disposed.
   *
   * @public
   */
  disposables = /* @__PURE__ */ new Set();
  /**
   * Whether the editor is disposed.
   *
   * @public
   */
  isDisposed = false;
  /** @internal */
  _tickManager;
  /**
   * A manager for the app's snapping feature.
   *
   * @public
   */
  snaps;
  /**
   * A manager for the any asynchronous events and making sure they're
   * cleaned up upon disposal.
   *
   * @public
   */
  timers;
  /**
   * A manager for the user and their preferences.
   *
   * @public
   */
  user;
  /**
   * A helper for measuring text.
   *
   * @public
   */
  textMeasure;
  /**
   * A manager for the editor's environment.
   *
   * @public
   */
  environment;
  /**
   * A manager for the editor's scribbles.
   *
   * @public
   */
  scribbles;
  /**
   * A manager for side effects and correct state enforcement. See {@link @tldraw/store#StoreSideEffects} for details.
   *
   * @public
   */
  sideEffects;
  /**
   * A manager for moving the camera when the mouse is at the edge of the screen.
   *
   * @public
   */
  edgeScrollManager;
  /**
   * A manager for ensuring correct focus. See FocusManager for details.
   *
   * @internal
   */
  focusManager;
  /**
   * The current HTML element containing the editor.
   *
   * @example
   * ```ts
   * const container = editor.getContainer()
   * ```
   *
   * @public
   */
  getContainer;
  /**
   * Dispose the editor.
   *
   * @public
   */
  dispose() {
    this.disposables.forEach((dispose) => dispose());
    this.disposables.clear();
    this.isDisposed = true;
  }
  /* ------------------- Shape Utils ------------------ */
  /**
   * A map of shape utility classes (TLShapeUtils) by shape type.
   *
   * @public
   */
  shapeUtils;
  styleProps;
  getShapeUtil(arg) {
    const type = typeof arg === "string" ? arg : arg.type;
    const shapeUtil = getOwnProperty(this.shapeUtils, type);
    assert(shapeUtil, `No shape util found for type "${type}"`);
    return shapeUtil;
  }
  /* ------------------- Binding Utils ------------------ */
  /**
   * A map of shape utility classes (TLShapeUtils) by shape type.
   *
   * @public
   */
  bindingUtils;
  getBindingUtil(arg) {
    const type = typeof arg === "string" ? arg : arg.type;
    const bindingUtil = getOwnProperty(this.bindingUtils, type);
    assert(bindingUtil, `No binding util found for type "${type}"`);
    return bindingUtil;
  }
  /* --------------------- History -------------------- */
  /**
   * A manager for the app's history.
   *
   * @readonly
   */
  history;
  /**
   * Undo to the last mark.
   *
   * @example
   * ```ts
   * editor.undo()
   * ```
   *
   * @public
   */
  undo() {
    this._flushEventsForTick(0);
    this.complete();
    this.history.undo();
    return this;
  }
  getCanUndo() {
    return this.history.getNumUndos() > 0;
  }
  /**
   * Redo to the next mark.
   *
   * @example
   * ```ts
   * editor.redo()
   * ```
   *
   * @public
   */
  redo() {
    this._flushEventsForTick(0);
    this.complete();
    this.history.redo();
    return this;
  }
  clearHistory() {
    this.history.clear();
    return this;
  }
  getCanRedo() {
    return this.history.getNumRedos() > 0;
  }
  /**
   * Create a new "mark", or stopping point, in the undo redo history. Creating a mark will clear
   * any redos.
   *
   * @example
   * ```ts
   * editor.mark()
   * editor.mark('flip shapes')
   * ```
   *
   * @param markId - The mark's id, usually the reason for adding the mark.
   *
   * @public
   */
  mark(markId) {
    this.history.mark(markId);
    return this;
  }
  /**
   * Squash the history to the given mark id.
   *
   * @example
   * ```ts
   * editor.mark('bump shapes')
   * // ... some changes
   * editor.squashToMark('bump shapes')
   * ```
   *
   * @param markId - The mark id to squash to.
   */
  squashToMark(markId) {
    this.history.squashToMark(markId);
    return this;
  }
  /**
   * Clear all marks in the undo stack back to the next mark.
   *
   * @example
   * ```ts
   * editor.bail()
   * ```
   *
   * @public
   */
  bail() {
    this.history.bail();
    return this;
  }
  /**
   * Clear all marks in the undo stack back to the mark with the provided mark id.
   *
   * @example
   * ```ts
   * editor.bailToMark('dragging')
   * ```
   *
   * @public
   */
  bailToMark(id) {
    this.history.bailToMark(id);
    return this;
  }
  _shouldIgnoreShapeLock = false;
  /**
   * Run a function in a transaction with optional options for context.
   * You can use the options to change the way that history is treated
   * or allow changes to locked shapes.
   *
   * @example
   * ```ts
   * // updating with
   * editor.run(() => {
   * 	editor.updateShape({ ...myShape, x: 100 })
   * }, { history: "ignore" })
   *
   * // forcing changes / deletions for locked shapes
   * editor.toggleLock([myShape])
   * editor.run(() => {
   * 	editor.updateShape({ ...myShape, x: 100 })
   * 	editor.deleteShape(myShape)
   * }, { ignoreShapeLock: true }, )
   * ```
   *
   * @param fn - The callback function to run.
   * @param opts - The options for the batch.
   *
   *
   * @public
   */
  run(fn, opts) {
    const previousIgnoreShapeLock = this._shouldIgnoreShapeLock;
    this._shouldIgnoreShapeLock = opts?.ignoreShapeLock ?? previousIgnoreShapeLock;
    try {
      this.history.batch(fn, opts);
    } finally {
      this._shouldIgnoreShapeLock = previousIgnoreShapeLock;
    }
    return this;
  }
  /**
   * @deprecated Use `Editor.run` instead.
   */
  batch(fn, opts) {
    return this.run(fn, opts);
  }
  /* --------------------- Errors --------------------- */
  /** @internal */
  annotateError(error, {
    origin,
    willCrashApp,
    tags,
    extras
  }) {
    const defaultAnnotations = this.createErrorAnnotations(origin, willCrashApp);
    annotateError(error, {
      tags: { ...defaultAnnotations.tags, ...tags },
      extras: { ...defaultAnnotations.extras, ...extras }
    });
    if (willCrashApp) {
      this.store.markAsPossiblyCorrupted();
    }
    return this;
  }
  /** @internal */
  createErrorAnnotations(origin, willCrashApp) {
    try {
      const editingShapeId = this.getEditingShapeId();
      return {
        tags: {
          origin,
          willCrashApp
        },
        extras: {
          activeStateNode: this.root.getPath(),
          selectedShapes: this.getSelectedShapes(),
          editingShape: editingShapeId ? this.getShape(editingShapeId) : void 0,
          inputs: this.inputs
        }
      };
    } catch {
      return {
        tags: {
          origin,
          willCrashApp
        },
        extras: {}
      };
    }
  }
  /** @internal */
  _crashingError = null;
  /**
   * We can't use an `atom` here because there's a chance that when `crashAndReportError` is called,
   * we're in a transaction that's about to be rolled back due to the same error we're currently
   * reporting.
   *
   * Instead, to listen to changes to this value, you need to listen to app's `crash` event.
   *
   * @internal
   */
  getCrashingError() {
    return this._crashingError;
  }
  /** @internal */
  crash(error) {
    this._crashingError = error;
    this.store.markAsPossiblyCorrupted();
    this.emit("crash", { error });
    return this;
  }
  getPath() {
    return this.root.getPath().split("root.")[1];
  }
  /**
   * Get whether a certain tool (or other state node) is currently active.
   *
   * @example
   * ```ts
   * editor.isIn('select')
   * editor.isIn('select.brushing')
   * ```
   *
   * @param path - The path of active states, separated by periods.
   *
   * @public
   */
  isIn(path) {
    const ids = path.split(".").reverse();
    let state = this.root;
    while (ids.length > 0) {
      const id = ids.pop();
      if (!id) return true;
      const current = state.getCurrent();
      if (current?.id === id) {
        if (ids.length === 0) return true;
        state = current;
        continue;
      } else return false;
    }
    return false;
  }
  /**
   * Get whether the state node is in any of the given active paths.
   *
   * @example
   * ```ts
   * state.isInAny('select', 'erase')
   * state.isInAny('select.brushing', 'erase.idle')
   * ```
   *
   * @public
   */
  isInAny(...paths) {
    return paths.some((path) => this.isIn(path));
  }
  /**
   * Set the selected tool.
   *
   * @example
   * ```ts
   * editor.setCurrentTool('hand')
   * editor.setCurrentTool('hand', { date: Date.now() })
   * ```
   *
   * @param id - The id of the tool to select.
   * @param info - Arbitrary data to pass along into the transition.
   *
   * @public
   */
  setCurrentTool(id, info = {}) {
    this.root.transition(id, info);
    return this;
  }
  getCurrentTool() {
    return this.root.getCurrent();
  }
  getCurrentToolId() {
    const currentTool = this.getCurrentTool();
    if (!currentTool) return "";
    return currentTool.getCurrentToolIdMask() ?? currentTool.id;
  }
  /**
   * Get a descendant by its path.
   *
   * @example
   * ```ts
   * state.getStateDescendant('select')
   * state.getStateDescendant('select.brushing')
   * ```
   *
   * @param path - The descendant's path of state ids, separated by periods.
   *
   * @public
   */
  getStateDescendant(path) {
    const ids = path.split(".").reverse();
    let state = this.root;
    while (ids.length > 0) {
      const id = ids.pop();
      if (!id) return state;
      const childState = state.children?.[id];
      if (!childState) return void 0;
      state = childState;
    }
    return state;
  }
  getDocumentSettings() {
    return this.store.get(TLDOCUMENT_ID);
  }
  /**
   * Update the global document settings that apply to all users.
   *
   * @public
   **/
  updateDocumentSettings(settings) {
    this.run(
      () => {
        this.store.put([{ ...this.getDocumentSettings(), ...settings }]);
      },
      { history: "ignore" }
    );
    return this;
  }
  getInstanceState() {
    return this.store.get(TLINSTANCE_ID);
  }
  /**
   * Update the instance's state.
   *
   * @param partial - A partial object to update the instance state with.
   *
   * @public
   */
  updateInstanceState(partial, historyOptions) {
    this._updateInstanceState(partial, { history: "ignore", ...historyOptions });
    if (partial.isChangingStyle !== void 0) {
      clearTimeout(this._isChangingStyleTimeout);
      if (partial.isChangingStyle === true) {
        this._isChangingStyleTimeout = this.timers.setTimeout(() => {
          this._updateInstanceState({ isChangingStyle: false }, { history: "ignore" });
        }, 2e3);
      }
    }
    return this;
  }
  /** @internal */
  _updateInstanceState = (partial, opts) => {
    this.run(() => {
      this.store.put([
        {
          ...this.getInstanceState(),
          ...partial
        }
      ]);
    }, opts);
  };
  /** @internal */
  _isChangingStyleTimeout = -1;
  getOpenMenus() {
    return this.getInstanceState().openMenus;
  }
  /**
   * Add an open menu.
   *
   * @example
   * ```ts
   * editor.addOpenMenu('menu-id')
   * ```
   *
   * @public
   */
  addOpenMenu(id) {
    const menus = new Set(this.getOpenMenus());
    if (!menus.has(id)) {
      menus.add(id);
      this.updateInstanceState({ openMenus: [...menus] });
    }
    return this;
  }
  /**
   * Delete an open menu.
   *
   * @example
   * ```ts
   * editor.deleteOpenMenu('menu-id')
   * ```
   *
   * @public
   */
  deleteOpenMenu(id) {
    const menus = new Set(this.getOpenMenus());
    if (menus.has(id)) {
      menus.delete(id);
      this.updateInstanceState({ openMenus: [...menus] });
    }
    return this;
  }
  /**
   * Clear all open menus.
   *
   * @example
   * ```ts
   * editor.clearOpenMenus()
   * ```
   *
   * @public
   */
  clearOpenMenus() {
    if (this.getOpenMenus().length) {
      this.updateInstanceState({ openMenus: [] });
    }
    return this;
  }
  getIsMenuOpen() {
    return this.getOpenMenus().length > 0;
  }
  /* --------------------- Cursor --------------------- */
  /**
   * Set the cursor.
   *
   * @param type - The cursor type.
   * @param rotation - The cursor rotation.
   *
   * @public
   */
  setCursor = (cursor) => {
    this.updateInstanceState({ cursor: { ...this.getInstanceState().cursor, ...cursor } });
    return this;
  };
  getPageStates() {
    return this._getPageStatesQuery().get();
  }
  _getPageStatesQuery() {
    return this.store.query.records("instance_page_state");
  }
  getCurrentPageState() {
    return this.store.get(this._getCurrentPageStateId());
  }
  _getCurrentPageStateId() {
    return InstancePageStateRecordType.createId(this.getCurrentPageId());
  }
  /**
   * Update this instance's page state.
   *
   * @example
   * ```ts
   * editor.updateCurrentPageState({ id: 'page1', editingShapeId: 'shape:123' })
   * ```
   *
   * @param partial - The partial of the page state object containing the changes.
   *
   * @public
   */
  updateCurrentPageState(partial) {
    this._updateCurrentPageState(partial);
    return this;
  }
  _updateCurrentPageState = (partial) => {
    this.store.update(partial.id ?? this.getCurrentPageState().id, (state) => ({
      ...state,
      ...partial
    }));
  };
  getSelectedShapeIds() {
    return this.getCurrentPageState().selectedShapeIds;
  }
  getSelectedShapes() {
    const { selectedShapeIds } = this.getCurrentPageState();
    return compact(selectedShapeIds.map((id) => this.store.get(id)));
  }
  /**
   * Select one or more shapes.
   *
   * @example
   * ```ts
   * editor.setSelectedShapes(['id1'])
   * editor.setSelectedShapes(['id1', 'id2'])
   * ```
   *
   * @param ids - The ids to select.
   *
   * @public
   */
  setSelectedShapes(shapes) {
    return this.run(
      () => {
        const ids = shapes.map((shape) => typeof shape === "string" ? shape : shape.id);
        const { selectedShapeIds: prevSelectedShapeIds } = this.getCurrentPageState();
        const prevSet = new Set(prevSelectedShapeIds);
        if (ids.length === prevSet.size && ids.every((id) => prevSet.has(id))) return null;
        this.store.put([{ ...this.getCurrentPageState(), selectedShapeIds: ids }]);
      },
      { history: "record-preserveRedoStack" }
    );
  }
  /**
   * Determine whether or not any of a shape's ancestors are selected.
   *
   * @param id - The id of the shape to check.
   *
   * @public
   */
  isAncestorSelected(shape) {
    const id = typeof shape === "string" ? shape : shape?.id ?? null;
    const _shape = this.getShape(id);
    if (!_shape) return false;
    const selectedShapeIds = this.getSelectedShapeIds();
    return !!this.findShapeAncestor(_shape, (parent) => selectedShapeIds.includes(parent.id));
  }
  /**
   * Select one or more shapes.
   *
   * @example
   * ```ts
   * editor.select('id1')
   * editor.select('id1', 'id2')
   * ```
   *
   * @param ids - The ids to select.
   *
   * @public
   */
  select(...shapes) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((shape) => shape.id);
    this.setSelectedShapes(ids);
    return this;
  }
  /**
   * Remove a shape from the existing set of selected shapes.
   *
   * @example
   * ```ts
   * editor.deselect(shape.id)
   * ```
   *
   * @public
   */
  deselect(...shapes) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((shape) => shape.id);
    const selectedShapeIds = this.getSelectedShapeIds();
    if (selectedShapeIds.length > 0 && ids.length > 0) {
      this.setSelectedShapes(selectedShapeIds.filter((id) => !ids.includes(id)));
    }
    return this;
  }
  /**
   * Select all direct children of the current page.
   *
   * @example
   * ```ts
   * editor.selectAll()
   * ```
   *
   * @public
   */
  selectAll() {
    const ids = this.getSortedChildIdsForParent(this.getCurrentPageId());
    if (ids.length <= 0) return this;
    this.setSelectedShapes(this._getUnlockedShapeIds(ids));
    return this;
  }
  /**
   * Clear the selection.
   *
   * @example
   * ```ts
   * editor.selectNone()
   * ```
   *
   * @public
   */
  selectNone() {
    if (this.getSelectedShapeIds().length > 0) {
      this.setSelectedShapes([]);
    }
    return this;
  }
  getOnlySelectedShapeId() {
    return this.getOnlySelectedShape()?.id ?? null;
  }
  getOnlySelectedShape() {
    const selectedShapes = this.getSelectedShapes();
    return selectedShapes.length === 1 ? selectedShapes[0] : null;
  }
  getSelectionPageBounds() {
    const selectedShapeIds = this.getCurrentPageState().selectedShapeIds;
    if (selectedShapeIds.length === 0) return null;
    return Box.Common(compact(selectedShapeIds.map((id) => this.getShapePageBounds(id))));
  }
  getSelectionRotation() {
    const selectedShapeIds = this.getSelectedShapeIds();
    let foundFirst = false;
    let rotation = 0;
    for (let i = 0, n = selectedShapeIds.length; i < n; i++) {
      const pageTransform = this.getShapePageTransform(selectedShapeIds[i]);
      if (!pageTransform) continue;
      if (foundFirst) {
        if (pageTransform.rotation() !== rotation) {
          return 0;
        }
      } else {
        foundFirst = true;
        rotation = pageTransform.rotation();
      }
    }
    return rotation;
  }
  getSelectionRotatedPageBounds() {
    const selectedShapeIds = this.getSelectedShapeIds();
    if (selectedShapeIds.length === 0) {
      return void 0;
    }
    const selectionRotation = this.getSelectionRotation();
    if (selectionRotation === 0) {
      return this.getSelectionPageBounds();
    }
    if (selectedShapeIds.length === 1) {
      const bounds = this.getShapeGeometry(selectedShapeIds[0]).bounds.clone();
      const pageTransform = this.getShapePageTransform(selectedShapeIds[0]);
      bounds.point = pageTransform.applyToPoint(bounds.point);
      return bounds;
    }
    const boxFromRotatedVertices = Box.FromPoints(
      this.getSelectedShapeIds().flatMap((id) => {
        const pageTransform = this.getShapePageTransform(id);
        if (!pageTransform) return [];
        return pageTransform.applyToPoints(this.getShapeGeometry(id).bounds.corners);
      }).map((p) => p.rot(-selectionRotation))
    );
    boxFromRotatedVertices.point = boxFromRotatedVertices.point.rot(selectionRotation);
    return boxFromRotatedVertices;
  }
  getSelectionRotatedScreenBounds() {
    const bounds = this.getSelectionRotatedPageBounds();
    if (!bounds) return void 0;
    const { x, y } = this.pageToScreen(bounds.point);
    const zoom = this.getZoomLevel();
    return new Box(x, y, bounds.width * zoom, bounds.height * zoom);
  }
  getFocusedGroupId() {
    return this.getCurrentPageState().focusedGroupId ?? this.getCurrentPageId();
  }
  getFocusedGroup() {
    const focusedGroupId = this.getFocusedGroupId();
    return focusedGroupId ? this.getShape(focusedGroupId) : void 0;
  }
  /**
   * Set the current focused group shape.
   *
   * @param shape - The group shape id (or group shape's id) to set as the focused group shape.
   *
   * @public
   */
  setFocusedGroup(shape) {
    const id = typeof shape === "string" ? shape : shape?.id ?? null;
    if (id !== null) {
      const shape2 = this.getShape(id);
      if (!shape2) {
        throw Error(`Editor.setFocusedGroup: Shape with id ${id} does not exist`);
      }
      if (!this.isShapeOfType(shape2, "group")) {
        throw Error(
          `Editor.setFocusedGroup: Cannot set focused group to shape of type ${shape2.type}`
        );
      }
    }
    if (id === this.getFocusedGroupId()) return this;
    return this.run(
      () => {
        this.store.update(this.getCurrentPageState().id, (s) => ({ ...s, focusedGroupId: id }));
      },
      { history: "record-preserveRedoStack" }
    );
  }
  /**
   * Exit the current focused group, moving up to the next parent group if there is one.
   *
   * @public
   */
  popFocusedGroupId() {
    const focusedGroup = this.getFocusedGroup();
    if (focusedGroup) {
      const match = this.findShapeAncestor(
        focusedGroup,
        (shape) => this.isShapeOfType(shape, "group")
      );
      this.setFocusedGroup(match?.id ?? null);
      this.select(focusedGroup.id);
    } else {
      this.setFocusedGroup(null);
      this.selectNone();
    }
    return this;
  }
  getEditingShapeId() {
    return this.getCurrentPageState().editingShapeId;
  }
  getEditingShape() {
    const editingShapeId = this.getEditingShapeId();
    return editingShapeId ? this.getShape(editingShapeId) : void 0;
  }
  /**
   * Set the current editing shape.
   *
   * @example
   * ```ts
   * editor.setEditingShape(myShape)
   * editor.setEditingShape(myShape.id)
   * ```
   *
   * @param shape - The shape (or shape id) to set as editing.
   *
   * @public
   */
  setEditingShape(shape) {
    const id = typeof shape === "string" ? shape : shape?.id ?? null;
    if (id !== this.getEditingShapeId()) {
      if (id) {
        const shape2 = this.getShape(id);
        if (shape2 && this.getShapeUtil(shape2).canEdit(shape2)) {
          this.run(
            () => {
              this._updateCurrentPageState({ editingShapeId: id });
            },
            { history: "ignore" }
          );
          return this;
        }
      }
      this.run(
        () => {
          this._updateCurrentPageState({ editingShapeId: null });
        },
        { history: "ignore" }
      );
    }
    return this;
  }
  getHoveredShapeId() {
    return this.getCurrentPageState().hoveredShapeId;
  }
  getHoveredShape() {
    const hoveredShapeId = this.getHoveredShapeId();
    return hoveredShapeId ? this.getShape(hoveredShapeId) : void 0;
  }
  /**
   * Set the editor's current hovered shape.
   *
   * @example
   * ```ts
   * editor.setHoveredShape(myShape)
   * editor.setHoveredShape(myShape.id)
   * ```
   *
   * @param shapes - The shape (or shape id) to set as hovered.
   *
   * @public
   */
  setHoveredShape(shape) {
    const id = typeof shape === "string" ? shape : shape?.id ?? null;
    if (id === this.getHoveredShapeId()) return this;
    this.run(
      () => {
        this.updateCurrentPageState({ hoveredShapeId: id });
      },
      { history: "ignore" }
    );
    return this;
  }
  getHintingShapeIds() {
    return this.getCurrentPageState().hintingShapeIds;
  }
  getHintingShape() {
    const hintingShapeIds = this.getHintingShapeIds();
    return compact(hintingShapeIds.map((id) => this.getShape(id)));
  }
  /**
   * Set the editor's current hinting shapes.
   *
   * @example
   * ```ts
   * editor.setHintingShapes([myShape])
   * editor.setHintingShapes([myShape.id])
   * ```
   *
   * @param shapes - The shapes (or shape ids) to set as hinting.
   *
   * @public
   */
  setHintingShapes(shapes) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((shape) => shape.id);
    this.run(
      () => {
        this._updateCurrentPageState({ hintingShapeIds: dedupe(ids) });
      },
      { history: "ignore" }
    );
    return this;
  }
  getErasingShapeIds() {
    return this.getCurrentPageState().erasingShapeIds;
  }
  getErasingShapes() {
    const erasingShapeIds = this.getErasingShapeIds();
    return compact(erasingShapeIds.map((id) => this.getShape(id)));
  }
  /**
   * Set the editor's current erasing shapes.
   *
   * @example
   * ```ts
   * editor.setErasingShapes([myShape])
   * editor.setErasingShapes([myShape.id])
   * ```
   *
   * @param shapes - The shapes (or shape ids) to set as hinting.
   *
   * @public
   */
  setErasingShapes(shapes) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((shape) => shape.id);
    ids.sort();
    const erasingShapeIds = this.getErasingShapeIds();
    this.run(
      () => {
        if (ids.length === erasingShapeIds.length) {
          for (let i = 0; i < ids.length; i++) {
            if (ids[i] !== erasingShapeIds[i]) {
              this._updateCurrentPageState({ erasingShapeIds: ids });
              break;
            }
          }
        } else {
          this._updateCurrentPageState({ erasingShapeIds: ids });
        }
      },
      { history: "ignore" }
    );
    return this;
  }
  // Cropping
  /**
   * The current cropping shape's id.
   *
   * @public
   */
  getCroppingShapeId() {
    return this.getCurrentPageState().croppingShapeId;
  }
  /**
   * Set the current cropping shape.
   *
   * @example
   * ```ts
   * editor.setCroppingShape(myShape)
   * editor.setCroppingShape(myShape.id)
   * ```
   *
   *
   * @param shape - The shape (or shape id) to set as cropping.
   *
   * @public
   */
  setCroppingShape(shape) {
    const id = typeof shape === "string" ? shape : shape?.id ?? null;
    if (id !== this.getCroppingShapeId()) {
      this.run(
        () => {
          if (!id) {
            this.updateCurrentPageState({ croppingShapeId: null });
          } else {
            const shape2 = this.getShape(id);
            const util = this.getShapeUtil(shape2);
            if (shape2 && util.canCrop(shape2)) {
              this.updateCurrentPageState({ croppingShapeId: id });
            }
          }
        },
        { history: "ignore" }
      );
    }
    return this;
  }
  _unsafe_getCameraId() {
    return CameraRecordType.createId(this.getCurrentPageId());
  }
  getCamera() {
    const baseCamera = this.store.get(this._unsafe_getCameraId());
    if (this._isLockedOnFollowingUser.get()) {
      const followingCamera = this.getCameraForFollowing();
      if (followingCamera) {
        return { ...baseCamera, ...followingCamera };
      }
    }
    return baseCamera;
  }
  getViewportPageBoundsForFollowing() {
    const followingUserId = this.getInstanceState().followingUserId;
    if (!followingUserId) return null;
    const leaderPresence = this.getCollaborators().find((c) => c.userId === followingUserId);
    if (!leaderPresence) return null;
    const { w: lw, h: lh } = leaderPresence.screenBounds;
    const { x: lx, y: ly, z: lz } = leaderPresence.camera;
    const theirViewport = new Box(-lx, -ly, lw / lz, lh / lz);
    const ourViewport = this.getViewportScreenBounds().clone();
    const ourAspectRatio = ourViewport.width / ourViewport.height;
    ourViewport.width = theirViewport.width;
    ourViewport.height = ourViewport.width / ourAspectRatio;
    if (ourViewport.height < theirViewport.height) {
      ourViewport.height = theirViewport.height;
      ourViewport.width = ourViewport.height * ourAspectRatio;
    }
    ourViewport.center = theirViewport.center;
    return ourViewport;
  }
  getCameraForFollowing() {
    const viewport = this.getViewportPageBoundsForFollowing();
    if (!viewport) return null;
    return {
      x: -viewport.x,
      y: -viewport.y,
      z: this.getViewportScreenBounds().w / viewport.width
    };
  }
  getZoomLevel() {
    return this.getCamera().z;
  }
  /**
   * Get the camera's initial or reset zoom level.
   *
   * @example
   * ```ts
   * editor.getInitialZoom()
   * ```
   *
   * @public */
  getInitialZoom() {
    const cameraOptions = this.getCameraOptions();
    if (!cameraOptions.constraints) return 1;
    if (cameraOptions.constraints.initialZoom === "default") return 1;
    const { zx, zy } = getCameraFitXFitY(this, cameraOptions);
    switch (cameraOptions.constraints.initialZoom) {
      case "fit-min": {
        return Math.max(zx, zy);
      }
      case "fit-max": {
        return Math.min(zx, zy);
      }
      case "fit-x": {
        return zx;
      }
      case "fit-y": {
        return zy;
      }
      case "fit-min-100": {
        return Math.min(1, Math.max(zx, zy));
      }
      case "fit-max-100": {
        return Math.min(1, Math.min(zx, zy));
      }
      case "fit-x-100": {
        return Math.min(1, zx);
      }
      case "fit-y-100": {
        return Math.min(1, zy);
      }
      default: {
        throw exhaustiveSwitchError(cameraOptions.constraints.initialZoom);
      }
    }
  }
  /**
   * Get the camera's base level for calculating actual zoom levels based on the zoom steps.
   *
   * @example
   * ```ts
   * editor.getBaseZoom()
   * ```
   *
   * @public */
  getBaseZoom() {
    const cameraOptions = this.getCameraOptions();
    if (!cameraOptions.constraints) return 1;
    if (cameraOptions.constraints.baseZoom === "default") return 1;
    const { zx, zy } = getCameraFitXFitY(this, cameraOptions);
    switch (cameraOptions.constraints.baseZoom) {
      case "fit-min": {
        return Math.max(zx, zy);
      }
      case "fit-max": {
        return Math.min(zx, zy);
      }
      case "fit-x": {
        return zx;
      }
      case "fit-y": {
        return zy;
      }
      case "fit-min-100": {
        return Math.min(1, Math.max(zx, zy));
      }
      case "fit-max-100": {
        return Math.min(1, Math.min(zx, zy));
      }
      case "fit-x-100": {
        return Math.min(1, zx);
      }
      case "fit-y-100": {
        return Math.min(1, zy);
      }
      default: {
        throw exhaustiveSwitchError(cameraOptions.constraints.baseZoom);
      }
    }
  }
  _cameraOptions = atom("camera options", DEFAULT_CAMERA_OPTIONS);
  /**
   * Get the current camera options.
   *
   * @example
   * ```ts
   * editor.getCameraOptions()
   * ```
   *
   *  @public */
  getCameraOptions() {
    return this._cameraOptions.get();
  }
  /**
   * Set the camera options. Changing the options won't immediately change the camera itself, so you may want to call `setCamera` after changing the options.
   *
   * @example
   * ```ts
   * editor.setCameraOptions(myCameraOptions)
   * editor.setCamera(editor.getCamera())
   * ```
   *
   * @param options - The camera options to set.
   *
   * @public */
  setCameraOptions(options) {
    const next = structuredClone({
      ...this._cameraOptions.__unsafe__getWithoutCapture(),
      ...options
    });
    if (next.zoomSteps?.length < 1) next.zoomSteps = [1];
    this._cameraOptions.set(next);
    return this;
  }
  /** @internal */
  getConstrainedCamera(point, opts) {
    const currentCamera = this.getCamera();
    let { x, y, z = currentCamera.z } = point;
    if (!opts?.force) {
      const cameraOptions = this.getCameraOptions();
      const zoomMin = cameraOptions.zoomSteps[0];
      const zoomMax = last(cameraOptions.zoomSteps);
      const vsb = this.getViewportScreenBounds();
      if (cameraOptions.constraints) {
        const { constraints } = cameraOptions;
        const py = Math.min(constraints.padding.y, vsb.w / 2);
        const px = Math.min(constraints.padding.x, vsb.h / 2);
        const bounds = Box.From(cameraOptions.constraints.bounds);
        const zx = (vsb.w - px * 2) / bounds.w;
        const zy = (vsb.h - py * 2) / bounds.h;
        const baseZoom = this.getBaseZoom();
        const maxZ = zoomMax * baseZoom;
        const minZ = zoomMin * baseZoom;
        if (opts?.reset) {
          z = this.getInitialZoom();
        }
        if (z < minZ || z > maxZ) {
          const { x: cx, y: cy, z: cz } = currentCamera;
          const cxA = -cx + vsb.w / cz / 2;
          const cyA = -cy + vsb.h / cz / 2;
          z = clamp(z, minZ, maxZ);
          const cxB = -cx + vsb.w / z / 2;
          const cyB = -cy + vsb.h / z / 2;
          x = cx + cxB - cxA;
          y = cy + cyB - cyA;
        }
        const minX = px / z - bounds.x;
        const minY = py / z - bounds.y;
        const freeW = (vsb.w - px * 2) / z - bounds.w;
        const freeH = (vsb.h - py * 2) / z - bounds.h;
        const originX = minX + freeW * constraints.origin.x;
        const originY = minY + freeH * constraints.origin.y;
        const behaviorX = typeof constraints.behavior === "string" ? constraints.behavior : constraints.behavior.x;
        const behaviorY = typeof constraints.behavior === "string" ? constraints.behavior : constraints.behavior.y;
        if (opts?.reset) {
          x = originX;
          y = originY;
        } else {
          switch (behaviorX) {
            case "fixed": {
              x = originX;
              break;
            }
            case "contain": {
              if (z < zx) x = originX;
              else x = clamp(x, minX + freeW, minX);
              break;
            }
            case "inside": {
              if (z < zx) x = clamp(x, minX, (vsb.w - px) / z - bounds.w);
              else x = clamp(x, minX + freeW, minX);
              break;
            }
            case "outside": {
              x = clamp(x, px / z - bounds.w, (vsb.w - px) / z);
              break;
            }
            case "free": {
              break;
            }
            default: {
              throw exhaustiveSwitchError(behaviorX);
            }
          }
          switch (behaviorY) {
            case "fixed": {
              y = originY;
              break;
            }
            case "contain": {
              if (z < zy) y = originY;
              else y = clamp(y, minY + freeH, minY);
              break;
            }
            case "inside": {
              if (z < zy) y = clamp(y, minY, (vsb.h - py) / z - bounds.h);
              else y = clamp(y, minY + freeH, minY);
              break;
            }
            case "outside": {
              y = clamp(y, py / z - bounds.h, (vsb.h - py) / z);
              break;
            }
            case "free": {
              break;
            }
            default: {
              throw exhaustiveSwitchError(behaviorY);
            }
          }
        }
      } else {
        if (z > zoomMax || z < zoomMin) {
          const { x: cx, y: cy, z: cz } = currentCamera;
          z = clamp(z, zoomMin, zoomMax);
          x = cx + (-cx + vsb.w / z / 2) - (-cx + vsb.w / cz / 2);
          y = cy + (-cy + vsb.h / z / 2) - (-cy + vsb.h / cz / 2);
        }
      }
    }
    return { x, y, z };
  }
  /** @internal */
  _setCamera(point, opts) {
    const currentCamera = this.getCamera();
    const { x, y, z } = this.getConstrainedCamera(point, opts);
    if (currentCamera.x === x && currentCamera.y === y && currentCamera.z === z) {
      return this;
    }
    transact(() => {
      const camera = { ...currentCamera, x, y, z };
      this.run(
        () => {
          this.store.put([camera]);
        },
        { history: "ignore" }
      );
      const { currentScreenPoint, currentPagePoint } = this.inputs;
      const { screenBounds } = this.store.unsafeGetWithoutCapture(TLINSTANCE_ID);
      if (currentScreenPoint.x / z - x !== currentPagePoint.x || currentScreenPoint.y / z - y !== currentPagePoint.y) {
        const event = {
          type: "pointer",
          target: "canvas",
          name: "pointer_move",
          // weird but true: we need to put the screen point back into client space
          point: Vec.AddXY(currentScreenPoint, screenBounds.x, screenBounds.y),
          pointerId: INTERNAL_POINTER_IDS.CAMERA_MOVE,
          ctrlKey: this.inputs.ctrlKey,
          altKey: this.inputs.altKey,
          shiftKey: this.inputs.shiftKey,
          button: 0,
          isPen: this.getInstanceState().isPenMode ?? false
        };
        if (opts?.immediate) {
          this._flushEventForTick(event);
        } else {
          this.dispatch(event);
        }
      }
      this._tickCameraState();
    });
    return this;
  }
  /**
   * Set the current camera.
   *
   * @example
   * ```ts
   * editor.setCamera({ x: 0, y: 0})
   * editor.setCamera({ x: 0, y: 0, z: 1.5})
   * editor.setCamera({ x: 0, y: 0, z: 1.5}, { animation: { duration: 1000, easing: (t) => t * t } })
   * ```
   *
   * @param point - The new camera position.
   * @param opts - The camera move options.
   *
   * @public
   */
  setCamera(point, opts) {
    const { isLocked } = this._cameraOptions.__unsafe__getWithoutCapture();
    if (isLocked && !opts?.force) return this;
    this.stopCameraAnimation();
    if (this.getInstanceState().followingUserId) {
      this.stopFollowingUser();
    }
    const _point = Vec.Cast(point);
    if (!Number.isFinite(_point.x)) _point.x = 0;
    if (!Number.isFinite(_point.y)) _point.y = 0;
    if (_point.z === void 0 || !Number.isFinite(_point.z)) point.z = this.getZoomLevel();
    const camera = this.getConstrainedCamera(_point, opts);
    if (opts?.animation) {
      const { width, height } = this.getViewportScreenBounds();
      this._animateToViewport(
        new Box(-camera.x, -camera.y, width / camera.z, height / camera.z),
        opts
      );
    } else {
      this._setCamera(camera, {
        ...opts,
        // we already did the constraining, so we don't need to do it again
        force: true
      });
    }
    return this;
  }
  /**
   * Center the camera on a point (in the current page space).
   *
   * @example
   * ```ts
   * editor.centerOnPoint({ x: 100, y: 100 })
   * editor.centerOnPoint({ x: 100, y: 100 }, { animation: { duration: 200 } })
   * ```
   *
   * @param point - The point in the current page space to center on.
   * @param animation - The camera move options.
   *
   * @public
   */
  centerOnPoint(point, opts) {
    const { isLocked } = this.getCameraOptions();
    if (isLocked && !opts?.force) return this;
    const { width: pw, height: ph } = this.getViewportPageBounds();
    this.setCamera(new Vec(-(point.x - pw / 2), -(point.y - ph / 2), this.getCamera().z), opts);
    return this;
  }
  /**
   * Zoom the camera to fit the current page's content in the viewport.
   *
   * @example
   * ```ts
   * editor.zoomToFit()
   * editor.zoomToFit({ animation: { duration: 200 } })
   * ```
   *
   * @param opts - The camera move options.
   *
   * @public
   */
  zoomToFit(opts) {
    const ids = [...this.getCurrentPageShapeIds()];
    if (ids.length <= 0) return this;
    const pageBounds = Box.Common(compact(ids.map((id) => this.getShapePageBounds(id))));
    this.zoomToBounds(pageBounds, opts);
    return this;
  }
  /**
   * Set the zoom back to 100%.
   *
   * @example
   * ```ts
   * editor.resetZoom()
   * editor.resetZoom(editor.getViewportScreenCenter(), { animation: { duration: 200 } })
   * editor.resetZoom(editor.getViewportScreenCenter(), { animation: { duration: 200 } })
   * ```
   *
   * @param point - The screen point to zoom out on. Defaults to the viewport screen center.
   * @param opts - The camera move options.
   *
   * @public
   */
  resetZoom(point = this.getViewportScreenCenter(), opts) {
    const { isLocked, constraints } = this.getCameraOptions();
    if (isLocked && !opts?.force) return this;
    const currentCamera = this.getCamera();
    const { x: cx, y: cy, z: cz } = currentCamera;
    const { x, y } = point;
    let z = 1;
    if (constraints) {
      const initialZoom = this.getInitialZoom();
      if (cz !== initialZoom) {
        z = initialZoom;
      }
    }
    this.setCamera(
      new Vec(cx + (x / z - x) - (x / cz - x), cy + (y / z - y) - (y / cz - y), z),
      opts
    );
    return this;
  }
  /**
   * Zoom the camera in.
   *
   * @example
   * ```ts
   * editor.zoomIn()
   * editor.zoomIn(editor.getViewportScreenCenter(), { animation: { duration: 200 } })
   * editor.zoomIn(editor.inputs.currentScreenPoint, { animation: { duration: 200 } })
   * ```
   *
   * @param point - The screen point to zoom in on. Defaults to the screen center
   * @param opts - The camera move options.
   *
   * @public
   */
  zoomIn(point = this.getViewportScreenCenter(), opts) {
    const { isLocked } = this.getCameraOptions();
    if (isLocked && !opts?.force) return this;
    const { x: cx, y: cy, z: cz } = this.getCamera();
    const { zoomSteps } = this.getCameraOptions();
    if (zoomSteps !== null && zoomSteps.length > 1) {
      const baseZoom = this.getBaseZoom();
      let zoom = last(zoomSteps) * baseZoom;
      for (let i = 1; i < zoomSteps.length; i++) {
        const z1 = zoomSteps[i - 1] * baseZoom;
        const z2 = zoomSteps[i] * baseZoom;
        if (z2 - cz <= (z2 - z1) / 2) continue;
        zoom = z2;
        break;
      }
      this.setCamera(
        new Vec(
          cx + (point.x / zoom - point.x) - (point.x / cz - point.x),
          cy + (point.y / zoom - point.y) - (point.y / cz - point.y),
          zoom
        ),
        opts
      );
    }
    return this;
  }
  /**
   * Zoom the camera out.
   *
   * @example
   * ```ts
   * editor.zoomOut()
   * editor.zoomOut(editor.getViewportScreenCenter(), { animation: { duration: 120 } })
   * editor.zoomOut(editor.inputs.currentScreenPoint, { animation: { duration: 120 } })
   * ```
   *
   * @param point - The point to zoom out on. Defaults to the viewport screen center.
   * @param opts - The camera move options.
   *
   * @public
   */
  zoomOut(point = this.getViewportScreenCenter(), opts) {
    const { isLocked } = this.getCameraOptions();
    if (isLocked && !opts?.force) return this;
    const { zoomSteps } = this.getCameraOptions();
    if (zoomSteps !== null && zoomSteps.length > 1) {
      const baseZoom = this.getBaseZoom();
      const { x: cx, y: cy, z: cz } = this.getCamera();
      let zoom = zoomSteps[0] * baseZoom;
      for (let i = zoomSteps.length - 1; i > 0; i--) {
        const z1 = zoomSteps[i - 1] * baseZoom;
        const z2 = zoomSteps[i] * baseZoom;
        if (z2 - cz >= (z2 - z1) / 2) continue;
        zoom = z1;
        break;
      }
      this.setCamera(
        new Vec(
          cx + (point.x / zoom - point.x) - (point.x / cz - point.x),
          cy + (point.y / zoom - point.y) - (point.y / cz - point.y),
          zoom
        ),
        opts
      );
    }
    return this;
  }
  /**
   * Zoom the camera to fit the current selection in the viewport.
   *
   * @example
   * ```ts
   * editor.zoomToSelection()
   * editor.zoomToSelection({ animation: { duration: 200 } })
   * ```
   *
   * @param animation - The camera move options.
   *
   * @public
   */
  zoomToSelection(opts) {
    const { isLocked } = this.getCameraOptions();
    if (isLocked && !opts?.force) return this;
    const selectionPageBounds = this.getSelectionPageBounds();
    if (selectionPageBounds) {
      this.zoomToBounds(selectionPageBounds, {
        targetZoom: Math.max(1, this.getZoomLevel()),
        ...opts
      });
    }
    return this;
  }
  /**
   * Zoom the camera to fit a bounding box (in the current page space).
   *
   * @example
   * ```ts
   * editor.zoomToBounds(myBounds)
   * editor.zoomToBounds(myBounds, { animation: { duration: 200 } })
   * editor.zoomToBounds(myBounds, { animation: { duration: 200 }, inset: 0, targetZoom: 1 })
   * ```
   *
   * @param bounds - The bounding box.
   * @param opts - The camera move options, target zoom, or custom inset amount.
   *
   * @public
   */
  zoomToBounds(bounds, opts) {
    const cameraOptions = this._cameraOptions.__unsafe__getWithoutCapture();
    if (cameraOptions.isLocked && !opts?.force) return this;
    const viewportScreenBounds = this.getViewportScreenBounds();
    const inset = opts?.inset ?? Math.min(ZOOM_TO_FIT_PADDING, viewportScreenBounds.width * 0.28);
    const baseZoom = this.getBaseZoom();
    const zoomMin = cameraOptions.zoomSteps[0];
    const zoomMax = last(cameraOptions.zoomSteps);
    let zoom = clamp(
      Math.min(
        (viewportScreenBounds.width - inset) / bounds.w,
        (viewportScreenBounds.height - inset) / bounds.h
      ),
      zoomMin * baseZoom,
      zoomMax * baseZoom
    );
    if (opts?.targetZoom !== void 0) {
      zoom = Math.min(opts.targetZoom, zoom);
    }
    this.setCamera(
      new Vec(
        -bounds.x + (viewportScreenBounds.width - bounds.w * zoom) / 2 / zoom,
        -bounds.y + (viewportScreenBounds.height - bounds.h * zoom) / 2 / zoom,
        zoom
      ),
      opts
    );
    return this;
  }
  /**
   * Stop the current camera animation, if any.
   *
   * @example
   * ```ts
   * editor.stopCameraAnimation()
   * ```
   *
   * @public
   */
  stopCameraAnimation() {
    this.emit("stop-camera-animation");
    return this;
  }
  /** @internal */
  _viewportAnimation = null;
  /** @internal */
  _animateViewport(ms) {
    if (!this._viewportAnimation) return;
    this._viewportAnimation.elapsed += ms;
    const { elapsed, easing, duration, start, end } = this._viewportAnimation;
    if (elapsed > duration) {
      this.off("tick", this._animateViewport);
      this._viewportAnimation = null;
      this._setCamera(new Vec(-end.x, -end.y, this.getViewportScreenBounds().width / end.width));
      return;
    }
    const remaining = duration - elapsed;
    const t = easing(1 - remaining / duration);
    const left = start.minX + (end.minX - start.minX) * t;
    const top = start.minY + (end.minY - start.minY) * t;
    const right = start.maxX + (end.maxX - start.maxX) * t;
    this._setCamera(new Vec(-left, -top, this.getViewportScreenBounds().width / (right - left)), {
      force: true
    });
  }
  /** @internal */
  _animateToViewport(targetViewportPage, opts = { animation: DEFAULT_ANIMATION_OPTIONS }) {
    const { animation, ...rest } = opts;
    if (!animation) return;
    const { duration = 0, easing = EASINGS.easeInOutCubic } = animation;
    const animationSpeed = this.user.getAnimationSpeed();
    const viewportPageBounds = this.getViewportPageBounds();
    this.stopCameraAnimation();
    if (this.getInstanceState().followingUserId) {
      this.stopFollowingUser();
    }
    if (duration === 0 || animationSpeed === 0) {
      return this._setCamera(
        new Vec(
          -targetViewportPage.x,
          -targetViewportPage.y,
          this.getViewportScreenBounds().width / targetViewportPage.width
        ),
        { ...rest }
      );
    }
    this._viewportAnimation = {
      elapsed: 0,
      duration: duration / animationSpeed,
      easing,
      start: viewportPageBounds.clone(),
      end: targetViewportPage.clone()
    };
    this.once("stop-camera-animation", () => {
      this.off("tick", this._animateViewport);
      this._viewportAnimation = null;
    });
    this.on("tick", this._animateViewport);
    return this;
  }
  /**
   * Slide the camera in a certain direction.
   *
   * @example
   * ```ts
   * editor.slideCamera({ speed: 1, direction: { x: 1, y: 0 }, friction: 0.1 })
   * ```
   *
   * @param opts - Options for the slide
   * @public
   */
  slideCamera(opts = {}) {
    const { isLocked } = this.getCameraOptions();
    if (isLocked && !opts?.force) return this;
    const animationSpeed = this.user.getAnimationSpeed();
    if (animationSpeed === 0) return this;
    this.stopCameraAnimation();
    const {
      speed,
      friction = this.options.cameraSlideFriction,
      direction,
      speedThreshold = 0.01
    } = opts;
    let currentSpeed = Math.min(speed, 1);
    const cancel = () => {
      this.off("tick", moveCamera);
      this.off("stop-camera-animation", cancel);
    };
    this.once("stop-camera-animation", cancel);
    const moveCamera = (elapsed) => {
      const { x: cx, y: cy, z: cz } = this.getCamera();
      const movementVec = Vec.Mul(direction, currentSpeed * elapsed / cz);
      currentSpeed *= 1 - friction;
      if (currentSpeed < speedThreshold) {
        cancel();
      } else {
        this._setCamera(new Vec(cx + movementVec.x, cy + movementVec.y, cz));
      }
    };
    this.on("tick", moveCamera);
    return this;
  }
  /**
   * Animate the camera to a user's cursor position. This also briefly show the user's cursor if it's not currently visible.
   *
   * @example
   * ```ts
   * editor.zoomToUser(myUserId)
   * editor.zoomToUser(myUserId, { animation: { duration: 200 } })
   * ```
   *
   * @param userId - The id of the user to animate to.
   * @param opts - The camera move options.
   * @public
   */
  zoomToUser(userId, opts = { animation: { duration: 500 } }) {
    const presence = this.getCollaborators().find((c) => c.userId === userId);
    if (!presence) return this;
    this.run(() => {
      if (this.getInstanceState().followingUserId !== null) {
        this.stopFollowingUser();
      }
      const isOnSamePage = presence.currentPageId === this.getCurrentPageId();
      if (!isOnSamePage) {
        this.setCurrentPage(presence.currentPageId);
      }
      if (opts && opts.animation && !isOnSamePage) {
        opts.animation = void 0;
      }
      this.centerOnPoint(presence.cursor, opts);
      const { highlightedUserIds } = this.getInstanceState();
      this.updateInstanceState({ highlightedUserIds: [...highlightedUserIds, userId] });
      this.timers.setTimeout(() => {
        const highlightedUserIds2 = [...this.getInstanceState().highlightedUserIds];
        const index = highlightedUserIds2.indexOf(userId);
        if (index < 0) return;
        highlightedUserIds2.splice(index, 1);
        this.updateInstanceState({ highlightedUserIds: highlightedUserIds2 });
      }, this.options.collaboratorIdleTimeoutMs);
    });
    return this;
  }
  // Viewport
  /** @internal */
  _willSetInitialBounds = true;
  /**
   * Update the viewport. The viewport will measure the size and screen position of its container
   * element. This should be done whenever the container's position on the screen changes.
   *
   * @example
   * ```ts
   * editor.updateViewportScreenBounds(new Box(0, 0, 1280, 1024))
   * editor.updateViewportScreenBounds(new Box(0, 0, 1280, 1024), true)
   * ```
   *
   * @param center - Whether to preserve the viewport page center as the viewport changes.
   *
   * @public
   */
  updateViewportScreenBounds(screenBounds, center = false) {
    screenBounds.width = Math.max(screenBounds.width, 1);
    screenBounds.height = Math.max(screenBounds.height, 1);
    const insets = [
      // top
      screenBounds.minY !== 0,
      // right
      !approximately(document.body.scrollWidth, screenBounds.maxX, 1),
      // bottom
      !approximately(document.body.scrollHeight, screenBounds.maxY, 1),
      // left
      screenBounds.minX !== 0
    ];
    const { screenBounds: prevScreenBounds, insets: prevInsets } = this.getInstanceState();
    if (screenBounds.equals(prevScreenBounds) && insets.every((v, i) => v === prevInsets[i])) {
      return this;
    }
    const { _willSetInitialBounds } = this;
    this._willSetInitialBounds = false;
    if (_willSetInitialBounds) {
      this.updateInstanceState({ screenBounds: screenBounds.toJson(), insets });
      this.setCamera(this.getCamera());
    } else {
      if (center && !this.getInstanceState().followingUserId) {
        const before = this.getViewportPageBounds().center;
        this.updateInstanceState({ screenBounds: screenBounds.toJson(), insets });
        this.centerOnPoint(before);
      } else {
        this.updateInstanceState({ screenBounds: screenBounds.toJson(), insets });
        this._setCamera(Vec.From({ ...this.getCamera() }));
      }
    }
    this._tickCameraState();
    return this;
  }
  getViewportScreenBounds() {
    const { x, y, w, h } = this.getInstanceState().screenBounds;
    return new Box(x, y, w, h);
  }
  getViewportScreenCenter() {
    const viewportScreenBounds = this.getViewportScreenBounds();
    return new Vec(
      viewportScreenBounds.midX - viewportScreenBounds.minX,
      viewportScreenBounds.midY - viewportScreenBounds.minY
    );
  }
  getViewportPageBounds() {
    const { w, h } = this.getViewportScreenBounds();
    const { x: cx, y: cy, z: cz } = this.getCamera();
    return new Box(-cx, -cy, w / cz, h / cz);
  }
  /**
   * Convert a point in screen space to a point in the current page space.
   *
   * @example
   * ```ts
   * editor.screenToPage({ x: 100, y: 100 })
   * ```
   *
   * @param point - The point in screen space.
   *
   * @public
   */
  screenToPage(point) {
    const { screenBounds } = this.store.unsafeGetWithoutCapture(TLINSTANCE_ID);
    const { x: cx, y: cy, z: cz = 1 } = this.getCamera();
    return new Vec(
      (point.x - screenBounds.x) / cz - cx,
      (point.y - screenBounds.y) / cz - cy,
      point.z ?? 0.5
    );
  }
  /**
   * Convert a point in the current page space to a point in current screen space.
   *
   * @example
   * ```ts
   * editor.pageToScreen({ x: 100, y: 100 })
   * ```
   *
   * @param point - The point in page space.
   *
   * @public
   */
  pageToScreen(point) {
    const { screenBounds } = this.store.unsafeGetWithoutCapture(TLINSTANCE_ID);
    const { x: cx, y: cy, z: cz = 1 } = this.getCamera();
    return new Vec(
      (point.x + cx) * cz + screenBounds.x,
      (point.y + cy) * cz + screenBounds.y,
      point.z ?? 0.5
    );
  }
  /**
   * Convert a point in the current page space to a point in current viewport space.
   *
   * @example
   * ```ts
   * editor.pageToViewport({ x: 100, y: 100 })
   * ```
   *
   * @param point - The point in page space.
   *
   * @public
   */
  pageToViewport(point) {
    const { x: cx, y: cy, z: cz = 1 } = this.getCamera();
    return new Vec((point.x + cx) * cz, (point.y + cy) * cz, point.z ?? 0.5);
  }
  _getCollaboratorsQuery() {
    return this.store.query.records("instance_presence", () => ({
      userId: { neq: this.user.getId() }
    }));
  }
  getCollaborators() {
    const allPresenceRecords = this._getCollaboratorsQuery().get();
    if (!allPresenceRecords.length) return EMPTY_ARRAY;
    const userIds = [...new Set(allPresenceRecords.map((c) => c.userId))].sort();
    return userIds.map((id) => {
      const latestPresence = allPresenceRecords.filter((c) => c.userId === id).sort((a, b) => b.lastActivityTimestamp - a.lastActivityTimestamp)[0];
      return latestPresence;
    });
  }
  getCollaboratorsOnCurrentPage() {
    const currentPageId = this.getCurrentPageId();
    return this.getCollaborators().filter((c) => c.currentPageId === currentPageId);
  }
  // Following
  // When we are 'locked on' to a user, our camera is derived from their camera.
  _isLockedOnFollowingUser = atom("isLockedOnFollowingUser", false);
  /**
   * Start viewport-following a user.
   *
   * @example
   * ```ts
   * editor.startFollowingUser(myUserId)
   * ```
   *
   * @param userId - The id of the user to follow.
   * @param opts - Options for starting to follow a user.
   *
   * @public
   */
  startFollowingUser(userId) {
    this.stopFollowingUser();
    const leaderPresences = this._getCollaboratorsQuery().get().filter((p) => p.userId === userId);
    if (!leaderPresences.length) {
      console.warn("User not found");
      return this;
    }
    const thisUserId = this.user.getId();
    if (!thisUserId) {
      console.warn("You should set the userId for the current instance before following a user");
    }
    if (leaderPresences.some((p) => p.followingUserId === thisUserId)) {
      return this;
    }
    const latestLeaderPresence = computed("latestLeaderPresence", () => {
      return this.getCollaborators().find((p) => p.userId === userId);
    });
    transact(() => {
      this.updateInstanceState({ followingUserId: userId }, { history: "ignore" });
      const dispose = react("update current page", () => {
        const leaderPresence = latestLeaderPresence.get();
        if (!leaderPresence) {
          this.stopFollowingUser();
          return;
        }
        if (leaderPresence.currentPageId !== this.getCurrentPageId() && this.getPage(leaderPresence.currentPageId)) {
          this.run(
            () => {
              this.store.put([
                { ...this.getInstanceState(), currentPageId: leaderPresence.currentPageId }
              ]);
              this._isLockedOnFollowingUser.set(true);
            },
            { history: "ignore" }
          );
        }
      });
      const cancel = () => {
        dispose();
        this._isLockedOnFollowingUser.set(false);
        this.off("frame", moveTowardsUser);
        this.off("stop-following", cancel);
      };
      const moveTowardsUser = () => {
        const leaderPresence = latestLeaderPresence.get();
        if (!leaderPresence) {
          this.stopFollowingUser();
          return;
        }
        if (this._isLockedOnFollowingUser.get()) return;
        const animationSpeed = this.user.getAnimationSpeed();
        if (animationSpeed === 0) {
          this._isLockedOnFollowingUser.set(true);
          return;
        }
        const targetViewport = this.getViewportPageBoundsForFollowing();
        if (!targetViewport) {
          this.stopFollowingUser();
          return;
        }
        const currentViewport = this.getViewportPageBounds();
        const diffX = Math.abs(targetViewport.minX - currentViewport.minX) + Math.abs(targetViewport.maxX - currentViewport.maxX);
        const diffY = Math.abs(targetViewport.minY - currentViewport.minY) + Math.abs(targetViewport.maxY - currentViewport.maxY);
        if (diffX < this.options.followChaseViewportSnap && diffY < this.options.followChaseViewportSnap) {
          this._isLockedOnFollowingUser.set(true);
          return;
        }
        const t = clamp(animationSpeed * 0.5, 0.1, 0.8);
        const nextViewport = new Box(
          lerp(currentViewport.minX, targetViewport.minX, t),
          lerp(currentViewport.minY, targetViewport.minY, t),
          lerp(currentViewport.width, targetViewport.width, t),
          lerp(currentViewport.height, targetViewport.height, t)
        );
        const nextCamera = new Vec(
          -nextViewport.x,
          -nextViewport.y,
          this.getViewportScreenBounds().width / nextViewport.width
        );
        this.stopCameraAnimation();
        this._setCamera(nextCamera);
      };
      this.once("stop-following", cancel);
      this.addListener("frame", moveTowardsUser);
      moveTowardsUser();
    });
    return this;
  }
  /**
   * Stop viewport-following a user.
   *
   * @example
   * ```ts
   * editor.stopFollowingUser()
   * ```
   * @public
   */
  stopFollowingUser() {
    this.run(
      () => {
        this.store.put([this.getCamera()]);
        this._isLockedOnFollowingUser.set(false);
        this.updateInstanceState({ followingUserId: null });
        this.emit("stop-following");
      },
      { history: "ignore" }
    );
    return this;
  }
  /** @internal */
  getUnorderedRenderingShapes(useEditorState) {
    const renderingShapes = [];
    let nextIndex = this.options.maxShapesPerPage * 2;
    let nextBackgroundIndex = this.options.maxShapesPerPage;
    const erasingShapeIds = this.getErasingShapeIds();
    const addShapeById = (id, opacity, isAncestorErasing) => {
      const shape = this.getShape(id);
      if (!shape) return;
      opacity *= shape.opacity;
      let isShapeErasing = false;
      const util = this.getShapeUtil(shape);
      if (useEditorState) {
        isShapeErasing = !isAncestorErasing && erasingShapeIds.includes(id);
        if (isShapeErasing) {
          opacity *= 0.32;
        }
      }
      renderingShapes.push({
        id,
        shape,
        util,
        index: nextIndex,
        backgroundIndex: nextBackgroundIndex,
        opacity
      });
      nextIndex += 1;
      nextBackgroundIndex += 1;
      const childIds = this.getSortedChildIdsForParent(id);
      if (!childIds.length) return;
      let backgroundIndexToRestore = null;
      if (util.providesBackgroundForChildren(shape)) {
        backgroundIndexToRestore = nextBackgroundIndex;
        nextBackgroundIndex = nextIndex;
        nextIndex += this.options.maxShapesPerPage;
      }
      for (const childId of childIds) {
        addShapeById(childId, opacity, isAncestorErasing || isShapeErasing);
      }
      if (backgroundIndexToRestore !== null) {
        nextBackgroundIndex = backgroundIndexToRestore;
      }
    };
    const pages = useEditorState ? [this.getCurrentPage()] : this.getPages();
    for (const page of pages) {
      for (const childId of this.getSortedChildIdsForParent(page.id)) {
        addShapeById(childId, 1, false);
      }
    }
    return renderingShapes;
  }
  // Camera state
  // Camera state does two things: first, it allows us to subscribe to whether
  // the camera is moving or not; and second, it allows us to update the rendering
  // shapes on the canvas. Changing the rendering shapes may cause shapes to
  // unmount / remount in the DOM, which is expensive; and computing visibility is
  // also expensive in large projects. For this reason, we use a second bounding
  // box just for rendering, and we only update after the camera stops moving.
  _cameraState = atom("camera state", "idle");
  _cameraStateTimeoutRemaining = 0;
  _decayCameraStateTimeout = (elapsed) => {
    this._cameraStateTimeoutRemaining -= elapsed;
    if (this._cameraStateTimeoutRemaining > 0) return;
    this.off("tick", this._decayCameraStateTimeout);
    this._cameraState.set("idle");
  };
  _tickCameraState = () => {
    this._cameraStateTimeoutRemaining = this.options.cameraMovingTimeoutMs;
    if (this._cameraState.__unsafe__getWithoutCapture() !== "idle") return;
    this._cameraState.set("moving");
    this.on("tick", this._decayCameraStateTimeout);
  };
  /**
   * Whether the camera is moving or idle.
   *
   * @example
   * ```ts
   * editor.getCameraState()
   * ```
   *
   * @public
   */
  getCameraState() {
    return this._cameraState.get();
  }
  getRenderingShapes() {
    const renderingShapes = this.getUnorderedRenderingShapes(true);
    return renderingShapes.sort(sortById);
  }
  _getAllPagesQuery() {
    return this.store.query.records("page");
  }
  getPages() {
    return this._getAllPagesQuery().get().sort(sortByIndex);
  }
  /**
   * The current page.
   *
   * @example
   * ```ts
   * editor.getCurrentPage()
   * ```
   *
   * @public
   */
  getCurrentPage() {
    return this.getPage(this.getCurrentPageId());
  }
  getCurrentPageId() {
    return this.getInstanceState().currentPageId;
  }
  /**
   * Get a page.
   *
   * @example
   * ```ts
   * editor.getPage(myPage.id)
   * editor.getPage(myPage)
   * ```
   *
   * @param page - The page (or page id) to get.
   *
   * @public
   */
  getPage(page) {
    return this.store.get(typeof page === "string" ? page : page.id);
  }
  /* @internal */
  _currentPageShapeIds;
  /**
   * An array of all of the shapes on the current page.
   *
   * @example
   * ```ts
   * editor.getCurrentPageIds()
   * ```
   *
   * @public
   */
  getCurrentPageShapeIds() {
    return this._currentPageShapeIds.get();
  }
  getCurrentPageShapeIdsSorted() {
    return Array.from(this.getCurrentPageShapeIds()).sort();
  }
  /**
   * Get the ids of shapes on a page.
   *
   * @example
   * ```ts
   * const idsOnPage1 = editor.getPageShapeIds('page1')
   * const idsOnPage2 = editor.getPageShapeIds(myPage2)
   * ```
   *
   * @param page - The page (or page id) to get.
   *
   * @public
   **/
  getPageShapeIds(page) {
    const pageId = typeof page === "string" ? page : page.id;
    const result = this.store.query.exec("shape", { parentId: { eq: pageId } });
    return this.getShapeAndDescendantIds(result.map((s) => s.id));
  }
  /**
   * Set the current page.
   *
   * @example
   * ```ts
   * editor.setCurrentPage('page1')
   * editor.setCurrentPage(myPage1)
   * ```
   *
   * @param page - The page (or page id) to set as the current page.
   *
   * @public
   */
  setCurrentPage(page) {
    const pageId = typeof page === "string" ? page : page.id;
    if (!this.store.has(pageId)) {
      console.error("Tried to set the current page id to a page that doesn't exist.");
      return this;
    }
    this.stopFollowingUser();
    this.complete();
    return this.run(
      () => {
        this.store.put([{ ...this.getInstanceState(), currentPageId: pageId }]);
      },
      { history: "record-preserveRedoStack" }
    );
  }
  /**
   * Update a page.
   *
   * @example
   * ```ts
   * editor.updatePage({ id: 'page2', name: 'Page 2' })
   * ```
   *
   * @param partial - The partial of the shape to update.
   *
   * @public
   */
  updatePage(partial) {
    if (this.getInstanceState().isReadonly) return this;
    const prev = this.getPage(partial.id);
    if (!prev) return this;
    return this.run(() => this.store.update(partial.id, (page) => ({ ...page, ...partial })));
  }
  /**
   * Create a page.
   *
   * @example
   * ```ts
   * editor.createPage(myPage)
   * editor.createPage({ name: 'Page 2' })
   * ```
   *
   * @param page - The page (or page partial) to create.
   *
   * @public
   */
  createPage(page) {
    this.run(() => {
      if (this.getInstanceState().isReadonly) return;
      if (this.getPages().length >= this.options.maxPages) return;
      const pages = this.getPages();
      const name = getIncrementedName(
        page.name ?? "Page 1",
        pages.map((p) => p.name)
      );
      let index = page.index;
      if (!index || pages.some((p) => p.index === index)) {
        index = getIndexAbove(pages[pages.length - 1].index);
      }
      const newPage = PageRecordType.create({
        meta: {},
        ...page,
        name,
        index
      });
      this.store.put([newPage]);
    });
    return this;
  }
  /**
   * Delete a page.
   *
   * @example
   * ```ts
   * editor.deletePage('page1')
   * ```
   *
   * @param id - The id of the page to delete.
   *
   * @public
   */
  deletePage(page) {
    const id = typeof page === "string" ? page : page.id;
    this.run(() => {
      if (this.getInstanceState().isReadonly) return;
      const pages = this.getPages();
      if (pages.length === 1) return;
      const deletedPage = this.getPage(id);
      if (!deletedPage) return;
      if (id === this.getCurrentPageId()) {
        const index = pages.findIndex((page2) => page2.id === id);
        const next = pages[index - 1] ?? pages[index + 1];
        this.setCurrentPage(next.id);
      }
      this.store.remove([deletedPage.id]);
    });
    return this;
  }
  /**
   * Duplicate a page.
   *
   * @param id - The id of the page to duplicate. Defaults to the current page.
   * @param createId - The id of the new page. Defaults to a new id.
   *
   * @public
   */
  duplicatePage(page, createId = PageRecordType.createId()) {
    if (this.getPages().length >= this.options.maxPages) return this;
    const id = typeof page === "string" ? page : page.id;
    const freshPage = this.getPage(id);
    if (!freshPage) return this;
    const prevCamera = { ...this.getCamera() };
    const content = this.getContentFromCurrentPage(this.getSortedChildIdsForParent(freshPage.id));
    this.run(() => {
      const pages = this.getPages();
      const index = getIndexBetween(freshPage.index, pages[pages.indexOf(freshPage) + 1]?.index);
      this.createPage({ name: freshPage.name + " Copy", id: createId, index });
      this.setCurrentPage(createId);
      this.setCamera(prevCamera);
      if (content) {
        return this.putContentOntoCurrentPage(content);
      }
    });
    return this;
  }
  /**
   * Rename a page.
   *
   * @example
   * ```ts
   * editor.renamePage('page1', 'My Page')
   * ```
   *
   * @param id - The id of the page to rename.
   * @param name - The new name.
   *
   * @public
   */
  renamePage(page, name) {
    const id = typeof page === "string" ? page : page.id;
    if (this.getInstanceState().isReadonly) return this;
    this.updatePage({ id, name });
    return this;
  }
  _getAllAssetsQuery() {
    return this.store.query.records("asset");
  }
  /**
   * Get all assets in the editor.
   *
   * @public
   */
  getAssets() {
    return this._getAllAssetsQuery().get();
  }
  /**
   * Create one or more assets.
   *
   * @example
   * ```ts
   * editor.createAssets([...myAssets])
   * ```
   *
   * @param assets - The assets to create.
   *
   * @public
   */
  createAssets(assets) {
    if (this.getInstanceState().isReadonly) return this;
    if (assets.length <= 0) return this;
    this.run(() => this.store.put(assets), { history: "ignore" });
    return this;
  }
  /**
   * Update one or more assets.
   *
   * @example
   * ```ts
   * editor.updateAssets([{ id: 'asset1', name: 'New name' }])
   * ```
   *
   * @param assets - The assets to update.
   *
   * @public
   */
  updateAssets(assets) {
    if (this.getInstanceState().isReadonly) return this;
    if (assets.length <= 0) return this;
    this.run(
      () => {
        this.store.put(
          assets.map((partial) => ({
            ...this.store.get(partial.id),
            ...partial
          }))
        );
      },
      { history: "ignore" }
    );
    return this;
  }
  /**
   * Delete one or more assets.
   *
   * @example
   * ```ts
   * editor.deleteAssets(['asset1', 'asset2'])
   * ```
   *
   * @param ids - The assets to delete.
   *
   * @public
   */
  deleteAssets(assets) {
    if (this.getInstanceState().isReadonly) return this;
    const ids = typeof assets[0] === "string" ? assets : assets.map((a) => a.id);
    if (ids.length <= 0) return this;
    this.run(() => this.store.remove(ids), { history: "ignore" });
    return this;
  }
  /**
   * Get an asset by its id.
   *
   * @example
   * ```ts
   * editor.getAsset('asset1')
   * ```
   *
   * @param asset - The asset (or asset id) to get.
   *
   * @public
   */
  getAsset(asset) {
    return this.store.get(typeof asset === "string" ? asset : asset.id);
  }
  async resolveAssetUrl(assetId, context) {
    if (!assetId) return null;
    const asset = this.getAsset(assetId);
    if (!asset) return null;
    const { screenScale = 1, shouldResolveToOriginal = false } = context;
    const zoomStepFunction = (zoom) => Math.pow(2, Math.ceil(Math.log2(zoom)));
    const steppedScreenScale = Math.max(0.125, zoomStepFunction(screenScale));
    const networkEffectiveType = "connection" in navigator ? navigator.connection.effectiveType : null;
    const dpr = this.getInstanceState().devicePixelRatio;
    return await this.store.props.assets.resolve(asset, {
      screenScale: screenScale || 1,
      steppedScreenScale,
      dpr,
      networkEffectiveType,
      shouldResolveToOriginal
    });
  }
  /**
   * Upload an asset to the store's asset service, returning a URL that can be used to resolve the
   * asset.
   */
  async uploadAsset(asset, file) {
    return await this.store.props.assets.upload(asset, file);
  }
  _getShapeGeometryCache() {
    return this.store.createComputedCache(
      "bounds",
      (shape) => this.getShapeUtil(shape).getGeometry(shape),
      (a, b) => a.props === b.props
    );
  }
  /**
   * Get the geometry of a shape.
   *
   * @example
   * ```ts
   * editor.getShapeGeometry(myShape)
   * editor.getShapeGeometry(myShapeId)
   * ```
   *
   * @param shape - The shape (or shape id) to get the geometry for.
   *
   * @public
   */
  getShapeGeometry(shape) {
    return this._getShapeGeometryCache().get(typeof shape === "string" ? shape : shape.id);
  }
  _getShapeHandlesCache() {
    return this.store.createComputedCache("handles", (shape) => {
      return this.getShapeUtil(shape).getHandles?.(shape);
    });
  }
  /**
   * Get the handles (if any) for a shape.
   *
   * @example
   * ```ts
   * editor.getShapeHandles(myShape)
   * editor.getShapeHandles(myShapeId)
   * ```
   *
   * @param shape - The shape (or shape id) to get the handles for.
   * @public
   */
  getShapeHandles(shape) {
    return this._getShapeHandlesCache().get(typeof shape === "string" ? shape : shape.id);
  }
  /**
   * Get the local transform for a shape as a matrix model. This transform reflects both its
   * translation (x, y) from from either its parent's top left corner, if the shape's parent is
   * another shape, or else from the 0,0 of the page, if the shape's parent is the page; and the
   * shape's rotation.
   *
   * @example
   * ```ts
   * editor.getShapeLocalTransform(myShape)
   * ```
   *
   * @param shape - The shape to get the local transform for.
   *
   * @public
   */
  getShapeLocalTransform(shape) {
    const id = typeof shape === "string" ? shape : shape.id;
    const freshShape = this.getShape(id);
    if (!freshShape) throw Error("Editor.getTransform: shape not found");
    return Mat.Identity().translate(freshShape.x, freshShape.y).rotate(freshShape.rotation);
  }
  _getShapePageTransformCache() {
    return this.store.createComputedCache("pageTransformCache", (shape) => {
      if (isPageId(shape.parentId)) {
        return this.getShapeLocalTransform(shape);
      }
      const parentTransform = this._getShapePageTransformCache().get(shape.parentId) ?? Mat.Identity();
      return Mat.Compose(parentTransform, this.getShapeLocalTransform(shape));
    });
  }
  /**
   * Get the local transform of a shape's parent as a matrix model.
   *
   * @example
   * ```ts
   * editor.getShapeParentTransform(myShape)
   * ```
   *
   * @param shape - The shape (or shape id) to get the parent transform for.
   *
   * @public
   */
  getShapeParentTransform(shape) {
    const id = typeof shape === "string" ? shape : shape.id;
    const freshShape = this.getShape(id);
    if (!freshShape || isPageId(freshShape.parentId)) return Mat.Identity();
    return this._getShapePageTransformCache().get(freshShape.parentId) ?? Mat.Identity();
  }
  /**
   * Get the transform of a shape in the current page space.
   *
   * @example
   * ```ts
   * editor.getShapePageTransform(myShape)
   * editor.getShapePageTransform(myShapeId)
   * ```
   *
   * @param shape - The shape (or shape id) to get the page transform for.
   *
   * @public
   */
  getShapePageTransform(shape) {
    const id = typeof shape === "string" ? shape : shape.id;
    return this._getShapePageTransformCache().get(id) ?? Mat.Identity();
  }
  _getShapePageBoundsCache() {
    return this.store.createComputedCache("pageBoundsCache", (shape) => {
      const pageTransform = this._getShapePageTransformCache().get(shape.id);
      if (!pageTransform) return new Box();
      const result = Box.FromPoints(
        Mat.applyToPoints(pageTransform, this.getShapeGeometry(shape).vertices)
      );
      return result;
    });
  }
  /**
   * Get the bounds of a shape in the current page space.
   *
   * @example
   * ```ts
   * editor.getShapePageBounds(myShape)
   * editor.getShapePageBounds(myShapeId)
   * ```
   *
   * @param shape - The shape (or shape id) to get the bounds for.
   *
   * @public
   */
  getShapePageBounds(shape) {
    return this._getShapePageBoundsCache().get(typeof shape === "string" ? shape : shape.id);
  }
  _getShapeClipPathCache() {
    return this.store.createComputedCache("clipPathCache", (shape) => {
      const pageMask = this._getShapeMaskCache().get(shape.id);
      if (!pageMask) return void 0;
      if (pageMask.length === 0) {
        return `polygon(0px 0px, 0px 0px, 0px 0px)`;
      }
      const pageTransform = this._getShapePageTransformCache().get(shape.id);
      if (!pageTransform) return void 0;
      const localMask = Mat.applyToPoints(Mat.Inverse(pageTransform), pageMask);
      return `polygon(${localMask.map((p) => `${p.x}px ${p.y}px`).join(",")})`;
    });
  }
  /**
   * Get the clip path for a shape.
   *
   * @example
   * ```ts
   * const clipPath = editor.getShapeClipPath(shape)
   * const clipPath = editor.getShapeClipPath(shape.id)
   * ```
   *
   * @param shape - The shape (or shape id) to get the clip path for.
   *
   * @returns The clip path or undefined.
   *
   * @public
   */
  getShapeClipPath(shape) {
    return this._getShapeClipPathCache().get(typeof shape === "string" ? shape : shape.id);
  }
  _getShapeMaskCache() {
    return this.store.createComputedCache("pageMaskCache", (shape) => {
      if (isPageId(shape.parentId)) return void 0;
      const frameAncestors = this.getShapeAncestors(shape.id).filter(
        (shape2) => this.isShapeOfType(shape2, "frame")
      );
      if (frameAncestors.length === 0) return void 0;
      const pageMask = frameAncestors.map(
        (s) => (
          // Apply the frame transform to the frame outline to get the frame outline in the current page space
          (this._getShapePageTransformCache().get(s.id).applyToPoints(this.getShapeGeometry(s).vertices))
        )
      ).reduce((acc, b) => {
        if (!(b && acc)) return void 0;
        const intersection = intersectPolygonPolygon(acc, b);
        if (intersection) {
          return intersection.map(Vec.Cast);
        }
        return [];
      });
      return pageMask;
    });
  }
  /**
   * Get the mask (in the current page space) for a shape.
   *
   * @example
   * ```ts
   * const pageMask = editor.getShapeMask(shape.id)
   * ```
   *
   * @param id - The id of the shape to get the mask for.
   *
   * @returns The mask for the shape.
   *
   * @public
   */
  getShapeMask(shape) {
    return this._getShapeMaskCache().get(typeof shape === "string" ? shape : shape.id);
  }
  /**
   * Get the bounds of a shape in the current page space, incorporating any masks. For example, if the
   * shape were the child of a frame and was half way out of the frame, the bounds would be the half
   * of the shape that was in the frame.
   *
   * @example
   * ```ts
   * editor.getShapeMaskedPageBounds(myShape)
   * editor.getShapeMaskedPageBounds(myShapeId)
   * ```
   *
   * @param shape - The shape to get the masked bounds for.
   *
   * @public
   */
  getShapeMaskedPageBounds(shape) {
    if (typeof shape !== "string") shape = shape.id;
    return this._getShapeMaskedPageBoundsCache().get(shape);
  }
  _getShapeMaskedPageBoundsCache() {
    return this.store.createComputedCache("shapeMaskedPageBoundsCache", (shape) => {
      const pageBounds = this._getShapePageBoundsCache().get(shape.id);
      if (!pageBounds) return;
      const pageMask = this._getShapeMaskCache().get(shape.id);
      if (pageMask) {
        if (pageMask.length === 0) return void 0;
        const { corners } = pageBounds;
        if (corners.every((p, i) => p && Vec.Equals(p, pageMask[i]))) return pageBounds.clone();
        const intersection = intersectPolygonPolygon(pageMask, corners);
        if (!intersection) return;
        return Box.FromPoints(intersection);
      }
      return pageBounds;
    });
  }
  /**
   * Get the ancestors of a shape.
   *
   * @example
   * ```ts
   * const ancestors = editor.getShapeAncestors(myShape)
   * const ancestors = editor.getShapeAncestors(myShapeId)
   * ```
   *
   * @param shape - The shape (or shape id) to get the ancestors for.
   *
   * @public
   */
  getShapeAncestors(shape, acc = []) {
    const id = typeof shape === "string" ? shape : shape.id;
    const freshShape = this.getShape(id);
    if (!freshShape) return acc;
    const parentId = freshShape.parentId;
    if (isPageId(parentId)) {
      acc.reverse();
      return acc;
    }
    const parent = this.store.get(parentId);
    if (!parent) return acc;
    acc.push(parent);
    return this.getShapeAncestors(parent, acc);
  }
  /**
   * Find the first ancestor matching the given predicate
   *
   * @example
   * ```ts
   * const ancestor = editor.findShapeAncestor(myShape)
   * const ancestor = editor.findShapeAncestor(myShape.id)
   * const ancestor = editor.findShapeAncestor(myShape.id, (shape) => shape.type === 'frame')
   * ```
   *
   * @param shape - The shape to check the ancestors for.
   *
   * @public
   */
  findShapeAncestor(shape, predicate) {
    const id = typeof shape === "string" ? shape : shape.id;
    const freshShape = this.getShape(id);
    if (!freshShape) return;
    const parentId = freshShape.parentId;
    if (isPageId(parentId)) return;
    const parent = this.getShape(parentId);
    if (!parent) return;
    return predicate(parent) ? parent : this.findShapeAncestor(parent, predicate);
  }
  /**
   * Returns true if the the given shape has the given ancestor.
   *
   * @param shape - The shape.
   * @param ancestorId - The id of the ancestor.
   *
   * @public
   */
  hasAncestor(shape, ancestorId) {
    const id = typeof shape === "string" ? shape : shape?.id;
    const freshShape = id && this.getShape(id);
    if (!freshShape) return false;
    if (freshShape.parentId === ancestorId) return true;
    return this.hasAncestor(this.getShapeParent(freshShape), ancestorId);
  }
  /**
   * Get the common ancestor of two or more shapes that matches a predicate.
   *
   * @param shapes - The shapes (or shape ids) to check.
   * @param predicate - The predicate to match.
   */
  findCommonAncestor(shapes, predicate) {
    if (shapes.length === 0) {
      return;
    }
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    const freshShapes = compact(ids.map((id) => this.getShape(id)));
    if (freshShapes.length === 1) {
      const parentId = freshShapes[0].parentId;
      if (isPageId(parentId)) {
        return;
      }
      return predicate ? this.findShapeAncestor(freshShapes[0], predicate)?.id : parentId;
    }
    const [nodeA, ...others] = freshShapes;
    let ancestor = this.getShapeParent(nodeA);
    while (ancestor) {
      if (predicate && !predicate(ancestor)) {
        ancestor = this.getShapeParent(ancestor);
        continue;
      }
      if (others.every((shape) => this.hasAncestor(shape, ancestor.id))) {
        return ancestor.id;
      }
      ancestor = this.getShapeParent(ancestor);
    }
    return void 0;
  }
  isShapeOrAncestorLocked(arg) {
    const shape = typeof arg === "string" ? this.getShape(arg) : arg;
    if (shape === void 0) return false;
    if (shape.isLocked) return true;
    return this.isShapeOrAncestorLocked(this.getShapeParent(shape));
  }
  _notVisibleShapes() {
    return notVisibleShapes(this);
  }
  getCulledShapes() {
    const notVisibleShapes2 = this._notVisibleShapes().get();
    const selectedShapeIds = this.getSelectedShapeIds();
    const editingId = this.getEditingShapeId();
    const culledShapes = new Set(notVisibleShapes2);
    if (editingId) {
      culledShapes.delete(editingId);
    }
    selectedShapeIds.forEach((id) => {
      culledShapes.delete(id);
    });
    return culledShapes;
  }
  getCurrentPageBounds() {
    let commonBounds;
    this.getCurrentPageShapeIdsSorted().forEach((shapeId) => {
      const bounds = this.getShapeMaskedPageBounds(shapeId);
      if (!bounds) return;
      if (!commonBounds) {
        commonBounds = bounds.clone();
      } else {
        commonBounds = commonBounds.expand(bounds);
      }
    });
    return commonBounds;
  }
  /**
   * Get the top-most selected shape at the given point, ignoring groups.
   *
   * @param point - The point to check.
   *
   * @returns The top-most selected shape at the given point, or undefined if there is no shape at the point.
   */
  getSelectedShapeAtPoint(point) {
    const selectedShapeIds = this.getSelectedShapeIds();
    return this.getCurrentPageShapesSorted().filter((shape) => shape.type !== "group" && selectedShapeIds.includes(shape.id)).reverse().find((shape) => this.isPointInShape(shape, point, { hitInside: true, margin: 0 }));
  }
  /**
   * Get the shape at the current point.
   *
   * @param point - The point to check.
   * @param opts - Options for the check: `hitInside` to check if the point is inside the shape, `margin` to check if the point is within a margin of the shape, `hitFrameInside` to check if the point is inside the frame, and `filter` to filter the shapes to check.
   *
   * @returns The shape at the given point, or undefined if there is no shape at the point.
   */
  getShapeAtPoint(point, opts = {}) {
    const zoomLevel = this.getZoomLevel();
    const viewportPageBounds = this.getViewportPageBounds();
    const {
      filter,
      margin = 0,
      hitLocked = false,
      hitLabels = false,
      hitInside = false,
      hitFrameInside = false
    } = opts;
    let inHollowSmallestArea = Infinity;
    let inHollowSmallestAreaHit = null;
    let inMarginClosestToEdgeDistance = Infinity;
    let inMarginClosestToEdgeHit = null;
    const shapesToCheck = (opts.renderingOnly ? this.getCurrentPageRenderingShapesSorted() : this.getCurrentPageShapesSorted()).filter((shape) => {
      if (shape.isLocked && !hitLocked || this.isShapeOfType(shape, "group")) return false;
      const pageMask = this.getShapeMask(shape);
      if (pageMask && !pointInPolygon(point, pageMask)) return false;
      if (filter) return filter(shape);
      return true;
    });
    for (let i = shapesToCheck.length - 1; i >= 0; i--) {
      const shape = shapesToCheck[i];
      const geometry = this.getShapeGeometry(shape);
      const isGroup = geometry instanceof Group2d;
      const pointInShapeSpace = this.getPointInShapeSpace(shape, point);
      if (this.isShapeOfType(shape, "arrow") || this.isShapeOfType(shape, "geo") && shape.props.fill === "none") {
        if (shape.props.text.trim()) {
          for (const childGeometry of geometry.children) {
            if (childGeometry.isLabel && childGeometry.isPointInBounds(pointInShapeSpace)) {
              return shape;
            }
          }
        }
      }
      if (this.isShapeOfType(shape, "frame")) {
        const distance2 = geometry.distanceToPoint(pointInShapeSpace, hitInside);
        if (Math.abs(distance2) <= margin) {
          return inMarginClosestToEdgeHit || shape;
        }
        if (geometry.hitTestPoint(pointInShapeSpace, 0, true)) {
          return inMarginClosestToEdgeHit || inHollowSmallestAreaHit || (hitFrameInside ? shape : void 0);
        }
        continue;
      }
      let distance;
      if (isGroup) {
        let minDistance = Infinity;
        for (const childGeometry of geometry.children) {
          if (childGeometry.isLabel && !hitLabels) continue;
          const tDistance = childGeometry.distanceToPoint(pointInShapeSpace, hitInside);
          if (tDistance < minDistance) {
            minDistance = tDistance;
          }
        }
        distance = minDistance;
      } else {
        if (margin === 0 && (geometry.bounds.w < 1 || geometry.bounds.h < 1)) {
          distance = geometry.distanceToPoint(pointInShapeSpace, hitInside);
        } else {
          if (geometry.bounds.containsPoint(pointInShapeSpace, margin)) {
            distance = geometry.distanceToPoint(pointInShapeSpace, hitInside);
          } else {
            distance = Infinity;
          }
        }
      }
      if (geometry.isClosed) {
        if (distance <= margin) {
          if (geometry.isFilled || isGroup && geometry.children[0].isFilled) {
            return inMarginClosestToEdgeHit || shape;
          } else {
            if (this.getShapePageBounds(shape).contains(viewportPageBounds)) continue;
            if (Math.abs(distance) < margin) {
              if (Math.abs(distance) < inMarginClosestToEdgeDistance) {
                inMarginClosestToEdgeDistance = Math.abs(distance);
                inMarginClosestToEdgeHit = shape;
              }
            } else if (!inMarginClosestToEdgeHit) {
              const { area } = geometry;
              if (area < inHollowSmallestArea) {
                inHollowSmallestArea = area;
                inHollowSmallestAreaHit = shape;
              }
            }
          }
        }
      } else {
        if (distance < this.options.hitTestMargin / zoomLevel) {
          return shape;
        }
      }
    }
    return inMarginClosestToEdgeHit || inHollowSmallestAreaHit || void 0;
  }
  /**
   * Get the shapes, if any, at a given page point.
   *
   * @example
   * ```ts
   * editor.getShapesAtPoint({ x: 100, y: 100 })
   * editor.getShapesAtPoint({ x: 100, y: 100 }, { hitInside: true, exact: true })
   * ```
   *
   * @param point - The page point to test.
   *
   * @public
   */
  getShapesAtPoint(point, opts = {}) {
    return this.getCurrentPageShapes().filter((shape) => this.isPointInShape(shape, point, opts));
  }
  /**
   * Test whether a point (in the current page space) will will a shape. This method takes into account masks,
   * such as when a shape is the child of a frame and is partially clipped by the frame.
   *
   * @example
   * ```ts
   * editor.isPointInShape({ x: 100, y: 100 }, myShape)
   * ```
   *
   * @param shape - The shape to test against.
   * @param point - The page point to test (in the current page space).
   * @param hitInside - Whether to count as a hit if the point is inside of a closed shape.
   *
   * @public
   */
  isPointInShape(shape, point, opts = {}) {
    const { hitInside = false, margin = 0 } = opts;
    const id = typeof shape === "string" ? shape : shape.id;
    const pageMask = this.getShapeMask(id);
    if (pageMask && !pointInPolygon(point, pageMask)) return false;
    return this.getShapeGeometry(id).hitTestPoint(
      this.getPointInShapeSpace(shape, point),
      margin,
      hitInside
    );
  }
  /**
   * Convert a point in the current page space to a point in the local space of a shape. For example, if a
   * shape's page point were `{ x: 100, y: 100 }`, a page point at `{ x: 110, y: 110 }` would be at
   * `{ x: 10, y: 10 }` in the shape's local space.
   *
   * @example
   * ```ts
   * editor.getPointInShapeSpace(myShape, { x: 100, y: 100 })
   * ```
   *
   * @param shape - The shape to get the point in the local space of.
   * @param point - The page point to get in the local space of the shape.
   *
   * @public
   */
  getPointInShapeSpace(shape, point) {
    const id = typeof shape === "string" ? shape : shape.id;
    return this._getShapePageTransformCache().get(id).clone().invert().applyToPoint(point);
  }
  /**
   * Convert a delta in the current page space to a point in the local space of a shape's parent.
   *
   * @example
   * ```ts
   * editor.getPointInParentSpace(myShape.id, { x: 100, y: 100 })
   * ```
   *
   * @param shape - The shape to get the point in the local space of.
   * @param point - The page point to get in the local space of the shape.
   *
   * @public
   */
  getPointInParentSpace(shape, point) {
    const id = typeof shape === "string" ? shape : shape.id;
    const freshShape = this.getShape(id);
    if (!freshShape) return new Vec(0, 0);
    if (isPageId(freshShape.parentId)) return Vec.From(point);
    const parentTransform = this.getShapePageTransform(freshShape.parentId);
    if (!parentTransform) return Vec.From(point);
    return parentTransform.clone().invert().applyToPoint(point);
  }
  getCurrentPageShapes() {
    return Array.from(this.getCurrentPageShapeIds(), (id) => this.store.get(id));
  }
  getCurrentPageShapesSorted() {
    const result = [];
    const topLevelShapes = this.getSortedChildIdsForParent(this.getCurrentPageId());
    for (let i = 0, n = topLevelShapes.length; i < n; i++) {
      pushShapeWithDescendants(this, topLevelShapes[i], result);
    }
    return result;
  }
  getCurrentPageRenderingShapesSorted() {
    const culledShapes = this.getCulledShapes();
    return this.getCurrentPageShapesSorted().filter(({ id }) => !culledShapes.has(id));
  }
  isShapeOfType(arg, type) {
    const shape = typeof arg === "string" ? this.getShape(arg) : arg;
    if (!shape) return false;
    return shape.type === type;
  }
  /**
   * Get a shape by its id.
   *
   * @example
   * ```ts
   * editor.getShape('box1')
   * ```
   *
   * @param id - The id of the shape to get.
   *
   * @public
   */
  getShape(shape) {
    const id = typeof shape === "string" ? shape : shape.id;
    if (!isShapeId(id)) return void 0;
    return this.store.get(id);
  }
  /**
   * Get the parent shape for a given shape. Returns undefined if the shape is the direct child of
   * the page.
   *
   * @example
   * ```ts
   * editor.getShapeParent(myShape)
   * ```
   *
   * @public
   */
  getShapeParent(shape) {
    const id = typeof shape === "string" ? shape : shape?.id;
    if (!id) return void 0;
    const freshShape = this.getShape(id);
    if (freshShape === void 0 || !isShapeId(freshShape.parentId)) return void 0;
    return this.store.get(freshShape.parentId);
  }
  /**
   * If siblingShape and targetShape are siblings, this returns targetShape. If targetShape has an
   * ancestor who is a sibling of siblingShape, this returns that ancestor. Otherwise, this returns
   * undefined.
   *
   * @internal
   */
  getShapeNearestSibling(siblingShape, targetShape) {
    if (!targetShape) {
      return void 0;
    }
    if (targetShape.parentId === siblingShape.parentId) {
      return targetShape;
    }
    const ancestor = this.findShapeAncestor(
      targetShape,
      (ancestor2) => ancestor2.parentId === siblingShape.parentId
    );
    return ancestor;
  }
  /**
   * Get whether the given shape is the descendant of the given page.
   *
   * @example
   * ```ts
   * editor.isShapeInPage(myShape)
   * editor.isShapeInPage(myShape, 'page1')
   * ```
   *
   * @param shape - The shape to check.
   * @param pageId - The id of the page to check against. Defaults to the current page.
   *
   * @public
   */
  isShapeInPage(shape, pageId = this.getCurrentPageId()) {
    const id = typeof shape === "string" ? shape : shape.id;
    const shapeToCheck = this.getShape(id);
    if (!shapeToCheck) return false;
    let shapeIsInPage = false;
    if (shapeToCheck.parentId === pageId) {
      shapeIsInPage = true;
    } else {
      let parent = this.getShape(shapeToCheck.parentId);
      isInPageSearch: while (parent) {
        if (parent.parentId === pageId) {
          shapeIsInPage = true;
          break isInPageSearch;
        }
        parent = this.getShape(parent.parentId);
      }
    }
    return shapeIsInPage;
  }
  /**
   * Get the id of the containing page for a given shape.
   *
   * @param shape - The shape to get the page id for.
   *
   * @returns The id of the page that contains the shape, or undefined if the shape is undefined.
   *
   * @public
   */
  getAncestorPageId(shape) {
    const id = typeof shape === "string" ? shape : shape?.id;
    const _shape = id && this.getShape(id);
    if (!_shape) return void 0;
    if (isPageId(_shape.parentId)) {
      return _shape.parentId;
    } else {
      return this.getAncestorPageId(this.getShape(_shape.parentId));
    }
  }
  // Parents and children
  /**
   * A cache of parents to children.
   *
   * @internal
   */
  _parentIdsToChildIds;
  /**
   * Reparent shapes to a new parent. This operation preserves the shape's current page positions /
   * rotations.
   *
   * @example
   * ```ts
   * editor.reparentShapes([box1, box2], 'frame1')
   * editor.reparentShapes([box1.id, box2.id], 'frame1')
   * editor.reparentShapes([box1.id, box2.id], 'frame1', 4)
   * ```
   *
   * @param shapes - The shapes (or shape ids) of the shapes to reparent.
   * @param parentId - The id of the new parent shape.
   * @param insertIndex - The index to insert the children.
   *
   * @public
   */
  reparentShapes(shapes, parentId, insertIndex) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (ids.length === 0) return this;
    const changes = [];
    const parentTransform = isPageId(parentId) ? Mat.Identity() : this.getShapePageTransform(parentId);
    const parentPageRotation = parentTransform.rotation();
    let indices = [];
    const sibs = compact(this.getSortedChildIdsForParent(parentId).map((id) => this.getShape(id)));
    if (insertIndex) {
      const sibWithInsertIndex = sibs.find((s) => s.index === insertIndex);
      if (sibWithInsertIndex) {
        const sibAbove = sibs[sibs.indexOf(sibWithInsertIndex) + 1];
        if (sibAbove) {
          indices = getIndicesBetween(insertIndex, sibAbove.index, ids.length);
        } else {
          indices = getIndicesAbove(insertIndex, ids.length);
        }
      } else {
        const sibAbove = sibs.sort(sortByIndex).find((s) => s.index > insertIndex);
        if (sibAbove) {
          indices = getIndicesBetween(insertIndex, sibAbove.index, ids.length);
        } else {
          indices = getIndicesAbove(insertIndex, ids.length);
        }
      }
    } else {
      const sib = sibs.length && sibs[sibs.length - 1];
      indices = sib ? getIndicesAbove(sib.index, ids.length) : getIndices(ids.length);
    }
    const invertedParentTransform = parentTransform.clone().invert();
    const shapesToReparent = compact(ids.map((id) => this.getShape(id)));
    this.run(
      () => {
        for (let i = 0; i < shapesToReparent.length; i++) {
          const shape = shapesToReparent[i];
          const pageTransform = this.getShapePageTransform(shape);
          if (!pageTransform) continue;
          const pagePoint = pageTransform.point();
          if (!pagePoint) continue;
          const newPoint = invertedParentTransform.applyToPoint(pagePoint);
          const newRotation = pageTransform.rotation() - parentPageRotation;
          changes.push({
            id: shape.id,
            type: shape.type,
            parentId,
            x: newPoint.x,
            y: newPoint.y,
            rotation: newRotation,
            index: indices[i]
          });
        }
        this.updateShapes(changes);
      },
      { ignoreShapeLock: true }
    );
    return this;
  }
  /**
   * Get the index above the highest child of a given parent.
   *
   * @param parentId - The id of the parent.
   *
   * @returns The index.
   *
   * @public
   */
  getHighestIndexForParent(parent) {
    const parentId = typeof parent === "string" ? parent : parent.id;
    const children = this._parentIdsToChildIds.get()[parentId];
    if (!children || children.length === 0) {
      return "a1";
    }
    const shape = this.getShape(children[children.length - 1]);
    return getIndexAbove(shape.index);
  }
  /**
   * Get an array of all the children of a shape.
   *
   * @example
   * ```ts
   * editor.getSortedChildIdsForParent('frame1')
   * ```
   *
   * @param parentId - The id of the parent shape.
   *
   * @public
   */
  getSortedChildIdsForParent(parent) {
    const parentId = typeof parent === "string" ? parent : parent.id;
    const ids = this._parentIdsToChildIds.get()[parentId];
    if (!ids) return EMPTY_ARRAY;
    return ids;
  }
  /**
   * Run a visitor function for all descendants of a shape.
   *
   * @example
   * ```ts
   * editor.visitDescendants('frame1', myCallback)
   * ```
   *
   * @param parentId - The id of the parent shape.
   * @param visitor - The visitor function.
   *
   * @public
   */
  visitDescendants(parent, visitor) {
    const parentId = typeof parent === "string" ? parent : parent.id;
    const children = this.getSortedChildIdsForParent(parentId);
    for (const id of children) {
      if (visitor(id) === false) continue;
      this.visitDescendants(id, visitor);
    }
    return this;
  }
  /**
   * Get the shape ids of all descendants of the given shapes (including the shapes themselves). IDs are returned in z-index order.
   *
   * @param ids - The ids of the shapes to get descendants of.
   *
   * @returns The descendant ids.
   *
   * @public
   */
  getShapeAndDescendantIds(ids) {
    const shapeIds = /* @__PURE__ */ new Set();
    for (const shape of ids.map((id) => this.getShape(id)).sort(sortByIndex)) {
      shapeIds.add(shape.id);
      this.visitDescendants(shape, (descendantId) => {
        shapeIds.add(descendantId);
      });
    }
    return shapeIds;
  }
  /**
   * Get the shape that some shapes should be dropped on at a given point.
   *
   * @param point - The point to find the parent for.
   * @param droppingShapes - The shapes that are being dropped.
   *
   * @returns The shape to drop on.
   *
   * @public
   */
  getDroppingOverShape(point, droppingShapes = []) {
    const currentPageShapesSorted = this.getCurrentPageShapesSorted();
    for (let i = currentPageShapesSorted.length - 1; i >= 0; i--) {
      const shape = currentPageShapesSorted[i];
      if (
        // don't allow dropping on selected shapes
        this.getSelectedShapeIds().includes(shape.id) || // only allow shapes that can receive children
        !this.getShapeUtil(shape).canDropShapes(shape, droppingShapes) || // don't allow dropping a shape on itself or one of it's children
        droppingShapes.find((s) => s.id === shape.id || this.hasAncestor(shape, s.id))
      ) {
        continue;
      }
      const maskedPageBounds = this.getShapeMaskedPageBounds(shape.id);
      if (maskedPageBounds && maskedPageBounds.containsPoint(point) && this.getShapeGeometry(shape).hitTestPoint(this.getPointInShapeSpace(shape, point), 0, true)) {
        return shape;
      }
    }
  }
  /**
   * Get the shape that should be selected when you click on a given shape, assuming there is
   * nothing already selected. It will not return anything higher than or including the current
   * focus layer.
   *
   * @param shape - The shape to get the outermost selectable shape for.
   * @param filter - A function to filter the selectable shapes.
   *
   * @returns The outermost selectable shape.
   *
   * @public
   */
  getOutermostSelectableShape(shape, filter) {
    const id = typeof shape === "string" ? shape : shape.id;
    const freshShape = this.getShape(id);
    let match = freshShape;
    let node = freshShape;
    const focusedGroup = this.getFocusedGroup();
    while (node) {
      if (this.isShapeOfType(node, "group") && focusedGroup?.id !== node.id && !this.hasAncestor(focusedGroup, node.id) && (filter?.(node) ?? true)) {
        match = node;
      } else if (focusedGroup?.id === node.id) {
        break;
      }
      node = this.getShapeParent(node);
    }
    return match;
  }
  _getBindingsIndexCache() {
    const index = bindingsIndex(this);
    return this.store.createComputedCache("bindingsIndex", (shape) => {
      return index.get().get(shape.id);
    });
  }
  /**
   * Get a binding from the store by its ID if it exists.
   */
  getBinding(id) {
    return this.store.get(id);
  }
  /**
   * Get all bindings of a certain type _from_ a particular shape. These are the bindings whose
   * `fromId` matched the shape's ID.
   */
  getBindingsFromShape(shape, type) {
    const id = typeof shape === "string" ? shape : shape.id;
    return this.getBindingsInvolvingShape(id).filter(
      (b) => b.fromId === id && b.type === type
    );
  }
  /**
   * Get all bindings of a certain type _to_ a particular shape. These are the bindings whose
   * `toId` matches the shape's ID.
   */
  getBindingsToShape(shape, type) {
    const id = typeof shape === "string" ? shape : shape.id;
    return this.getBindingsInvolvingShape(id).filter(
      (b) => b.toId === id && b.type === type
    );
  }
  /**
   * Get all bindings involving a particular shape. This includes bindings where the shape is the
   * `fromId` or `toId`. If a type is provided, only bindings of that type are returned.
   */
  getBindingsInvolvingShape(shape, type) {
    const id = typeof shape === "string" ? shape : shape.id;
    const result = this._getBindingsIndexCache().get(id) ?? EMPTY_ARRAY;
    if (!type) return result;
    return result.filter((b) => b.type === type);
  }
  /**
   * Create bindings from a list of partial bindings. You can omit the ID and most props of a
   * binding, but the `type`, `toId`, and `fromId` must all be provided.
   */
  createBindings(partials) {
    const bindings = [];
    for (const partial of partials) {
      const fromShape = this.getShape(partial.fromId);
      const toShape = this.getShape(partial.toId);
      if (!fromShape || !toShape) continue;
      if (!this.canBindShapes({ fromShape, toShape, binding: partial })) continue;
      const util = this.getBindingUtil(partial.type);
      const defaultProps = util.getDefaultProps();
      const binding = this.store.schema.types.binding.create({
        ...partial,
        id: partial.id ?? createBindingId(),
        props: {
          ...defaultProps,
          ...partial.props
        }
      });
      bindings.push(binding);
    }
    this.store.put(bindings);
    return this;
  }
  /**
   * Create a single binding from a partial. You can omit the ID and most props of a binding, but
   * the `type`, `toId`, and `fromId` must all be provided.
   */
  createBinding(partial) {
    return this.createBindings([partial]);
  }
  /**
   * Update bindings from a list of partial bindings. Each partial must include an ID, which will
   * be used to match the binding to it's existing record. If there is no existing record, that
   * binding is skipped. The changes from the partial are merged into the existing record.
   */
  updateBindings(partials) {
    const updated = [];
    for (const partial of partials) {
      if (!partial) continue;
      const current = this.getBinding(partial.id);
      if (!current) continue;
      const updatedBinding = applyPartialToRecordWithProps(current, partial);
      if (updatedBinding === current) continue;
      const fromShape = this.getShape(updatedBinding.fromId);
      const toShape = this.getShape(updatedBinding.toId);
      if (!fromShape || !toShape) continue;
      if (!this.canBindShapes({ fromShape, toShape, binding: updatedBinding })) continue;
      updated.push(updatedBinding);
    }
    this.store.put(updated);
    return this;
  }
  /**
   * Update a binding from a partial binding. Each partial must include an ID, which will be used
   * to match the binding to it's existing record. If there is no existing record, that binding is
   * skipped. The changes from the partial are merged into the existing record.
   */
  updateBinding(partial) {
    return this.updateBindings([partial]);
  }
  /**
   * Delete several bindings by their IDs. If a binding ID doesn't exist, it's ignored.
   */
  deleteBindings(bindings, { isolateShapes = false } = {}) {
    const ids = bindings.map((binding) => typeof binding === "string" ? binding : binding.id);
    if (isolateShapes) {
      this.store.atomic(() => {
        for (const id of ids) {
          const binding = this.getBinding(id);
          if (!binding) continue;
          const util = this.getBindingUtil(binding);
          util.onBeforeIsolateFromShape?.({ binding, removedShape: this.getShape(binding.toId) });
          util.onBeforeIsolateToShape?.({ binding, removedShape: this.getShape(binding.fromId) });
          this.store.remove([id]);
        }
      });
    } else {
      this.store.remove(ids);
    }
    return this;
  }
  /**
   * Delete a binding by its ID. If the binding doesn't exist, it's ignored.
   */
  deleteBinding(binding, opts) {
    return this.deleteBindings([binding], opts);
  }
  canBindShapes({
    fromShape,
    toShape,
    binding
  }) {
    const fromShapeType = typeof fromShape === "string" ? fromShape : fromShape.type;
    const toShapeType = typeof toShape === "string" ? toShape : toShape.type;
    const bindingType = typeof binding === "string" ? binding : binding.type;
    const canBindOpts = { fromShapeType, toShapeType, bindingType };
    if (fromShapeType === toShapeType) {
      return this.getShapeUtil(fromShapeType).canBind(canBindOpts);
    }
    return this.getShapeUtil(fromShapeType).canBind(canBindOpts) && this.getShapeUtil(toShapeType).canBind(canBindOpts);
  }
  /* -------------------- Commands -------------------- */
  /**
   * Rotate shapes by a delta in radians.
   * Note: Currently, this assumes that the shapes are your currently selected shapes.
   *
   * @example
   * ```ts
   * editor.rotateShapesBy(editor.getSelectedShapeIds(), Math.PI)
   * editor.rotateShapesBy(editor.getSelectedShapeIds(), Math.PI / 2)
   * ```
   *
   * @param shapes - The shapes (or shape ids) of the shapes to move.
   * @param delta - The delta in radians to apply to the selection rotation.
   */
  rotateShapesBy(shapes, delta) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (ids.length <= 0) return this;
    const snapshot = getRotationSnapshot({ editor: this });
    if (!snapshot) return this;
    applyRotationToSnapshotShapes({ delta, snapshot, editor: this, stage: "one-off" });
    return this;
  }
  getChangesToTranslateShape(initialShape, newShapeCoords) {
    let workingShape = initialShape;
    const util = this.getShapeUtil(initialShape);
    workingShape = applyPartialToRecordWithProps(
      workingShape,
      util.onTranslateStart?.(workingShape) ?? void 0
    );
    workingShape = applyPartialToRecordWithProps(workingShape, {
      id: initialShape.id,
      type: initialShape.type,
      x: newShapeCoords.x,
      y: newShapeCoords.y
    });
    workingShape = applyPartialToRecordWithProps(
      workingShape,
      util.onTranslate?.(initialShape, workingShape) ?? void 0
    );
    workingShape = applyPartialToRecordWithProps(
      workingShape,
      util.onTranslateEnd?.(initialShape, workingShape) ?? void 0
    );
    return workingShape;
  }
  /**
   * Move shapes by a delta.
   *
   * @example
   * ```ts
   * editor.nudgeShapes(['box1', 'box2'], { x: 8, y: 8 })
   * ```
   *
   * @param shapes - The shapes (or shape ids) to move.
   * @param direction - The direction in which to move the shapes.
   * @param historyOptions - The history options for the change.
   */
  nudgeShapes(shapes, offset) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (ids.length <= 0) return this;
    const changes = [];
    for (const id of ids) {
      const shape = this.getShape(id);
      const localDelta = Vec.From(offset);
      const parentTransform = this.getShapeParentTransform(shape);
      if (parentTransform) localDelta.rot(-parentTransform.rotation());
      changes.push(this.getChangesToTranslateShape(shape, localDelta.add(shape)));
    }
    this.updateShapes(changes);
    return this;
  }
  /**
   * Duplicate shapes.
   *
   * @example
   * ```ts
   * editor.duplicateShapes(['box1', 'box2'], { x: 8, y: 8 })
   * editor.duplicateShapes(editor.getSelectedShapes(), { x: 8, y: 8 })
   * ```
   *
   * @param shapes - The shapes (or shape ids) to duplicate.
   * @param offset - The offset (in pixels) to apply to the duplicated shapes.
   *
   * @public
   */
  duplicateShapes(shapes, offset) {
    this.run(() => {
      const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
      if (ids.length <= 0) return this;
      const initialIds = new Set(ids);
      const shapeIdSet = this.getShapeAndDescendantIds(ids);
      const orderedShapeIds = [...shapeIdSet].reverse();
      const shapeIds = /* @__PURE__ */ new Map();
      for (const shapeId of shapeIdSet) {
        shapeIds.set(shapeId, createShapeId());
      }
      const { shapesToCreate, bindingsToCreate } = withIsolatedShapes(
        this,
        shapeIdSet,
        (bindingIdsToMaintain) => {
          const bindingsToCreate2 = [];
          for (const originalId of bindingIdsToMaintain) {
            const originalBinding = this.getBinding(originalId);
            if (!originalBinding) continue;
            const duplicatedId = createBindingId();
            bindingsToCreate2.push({
              ...originalBinding,
              id: duplicatedId,
              fromId: assertExists(shapeIds.get(originalBinding.fromId)),
              toId: assertExists(shapeIds.get(originalBinding.toId))
            });
          }
          const shapesToCreate2 = [];
          for (const originalId of orderedShapeIds) {
            const duplicatedId = assertExists(shapeIds.get(originalId));
            const originalShape = this.getShape(originalId);
            if (!originalShape) continue;
            let ox = 0;
            let oy = 0;
            if (offset && initialIds.has(originalId)) {
              const parentTransform = this.getShapeParentTransform(originalShape);
              const vec = new Vec(offset.x, offset.y).rot(-parentTransform.rotation());
              ox = vec.x;
              oy = vec.y;
            }
            const parentId = originalShape.parentId;
            const siblings = this.getSortedChildIdsForParent(parentId);
            const currentIndex = siblings.indexOf(originalShape.id);
            const siblingAboveId = siblings[currentIndex + 1];
            const siblingAbove = siblingAboveId ? this.getShape(siblingAboveId) : null;
            const index = siblingAbove ? getIndexBetween(originalShape.index, siblingAbove.index) : getIndexAbove(originalShape.index);
            shapesToCreate2.push({
              ...originalShape,
              id: duplicatedId,
              x: originalShape.x + ox,
              y: originalShape.y + oy,
              index,
              parentId: shapeIds.get(originalShape.parentId) ?? originalShape.parentId
            });
          }
          return { shapesToCreate: shapesToCreate2, bindingsToCreate: bindingsToCreate2 };
        }
      );
      const maxShapesReached = shapesToCreate.length + this.getCurrentPageShapeIds().size > this.options.maxShapesPerPage;
      if (maxShapesReached) {
        alertMaxShapes(this);
        return;
      }
      this.createShapes(shapesToCreate);
      this.createBindings(bindingsToCreate);
      this.setSelectedShapes(compact(ids.map((id) => shapeIds.get(id))));
      if (offset !== void 0) {
        const selectionPageBounds = this.getSelectionPageBounds();
        const viewportPageBounds = this.getViewportPageBounds();
        if (selectionPageBounds && !viewportPageBounds.contains(selectionPageBounds)) {
          this.centerOnPoint(selectionPageBounds.center, {
            animation: { duration: this.options.animationMediumMs }
          });
        }
      }
    });
    return this;
  }
  /**
   * Move shapes to page.
   *
   * @example
   * ```ts
   * editor.moveShapesToPage(['box1', 'box2'], 'page1')
   * ```
   *
   * @param shapes - The shapes (or shape ids) of the shapes to move.
   * @param pageId - The id of the page where the shapes will be moved.
   *
   * @public
   */
  moveShapesToPage(shapes, pageId) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (ids.length === 0) return this;
    if (this.getInstanceState().isReadonly) return this;
    const currentPageId = this.getCurrentPageId();
    if (pageId === currentPageId) return this;
    if (!this.store.has(pageId)) return this;
    const content = this.getContentFromCurrentPage(ids);
    if (!content) return this;
    if (this.getPageShapeIds(pageId).size + content.shapes.length > this.options.maxShapesPerPage) {
      alertMaxShapes(this, pageId);
      return this;
    }
    const fromPageZ = this.getCamera().z;
    this.run(() => {
      this.deleteShapes(ids);
      this.setCurrentPage(pageId);
      this.setFocusedGroup(null);
      this.selectNone();
      this.putContentOntoCurrentPage(content, {
        select: true,
        preserveIds: true,
        preservePosition: true
      });
      this.setCamera({ ...this.getCamera(), z: fromPageZ });
      this.centerOnPoint(this.getSelectionRotatedPageBounds().center);
    });
    return this;
  }
  /**
   * Toggle the lock state of one or more shapes. If there is a mix of locked and unlocked shapes, all shapes will be locked.
   *
   * @param shapes - The shapes (or shape ids) to toggle.
   *
   * @public
   */
  toggleLock(shapes) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (this.getInstanceState().isReadonly || ids.length === 0) return this;
    let allLocked = true, allUnlocked = true;
    const shapesToToggle = [];
    for (const id of ids) {
      const shape = this.getShape(id);
      if (shape) {
        shapesToToggle.push(shape);
        if (shape.isLocked) {
          allUnlocked = false;
        } else {
          allLocked = false;
        }
      }
    }
    this.run(() => {
      if (allUnlocked) {
        this.updateShapes(
          shapesToToggle.map((shape) => ({ id: shape.id, type: shape.type, isLocked: true }))
        );
        this.setSelectedShapes([]);
      } else if (allLocked) {
        this.updateShapes(
          shapesToToggle.map((shape) => ({ id: shape.id, type: shape.type, isLocked: false }))
        );
      } else {
        this.updateShapes(
          shapesToToggle.map((shape) => ({ id: shape.id, type: shape.type, isLocked: true }))
        );
      }
    });
    return this;
  }
  /**
   * Send shapes to the back of the page's object list.
   *
   * @example
   * ```ts
   * editor.sendToBack(['id1', 'id2'])
   * editor.sendToBack(box1, box2)
   * ```
   *
   * @param shapes - The shapes (or shape ids) to move.
   *
   * @public
   */
  sendToBack(shapes) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    const changes = getReorderingShapesChanges(this, "toBack", ids);
    if (changes) this.updateShapes(changes);
    return this;
  }
  /**
   * Send shapes backward in the page's object list.
   *
   * @example
   * ```ts
   * editor.sendBackward(['id1', 'id2'])
   * editor.sendBackward([box1, box2])
   * ```
   *
   * @param shapes - The shapes (or shape ids) to move.
   *
   * @public
   */
  sendBackward(shapes) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    const changes = getReorderingShapesChanges(this, "backward", ids);
    if (changes) this.updateShapes(changes);
    return this;
  }
  /**
   * Bring shapes forward in the page's object list.
   *
   * @example
   * ```ts
   * editor.bringForward(['id1', 'id2'])
   * editor.bringForward(box1,  box2)
   * ```
   *
   * @param shapes - The shapes (or shape ids) to move.
   *
   * @public
   */
  bringForward(shapes) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    const changes = getReorderingShapesChanges(this, "forward", ids);
    if (changes) this.updateShapes(changes);
    return this;
  }
  /**
   * Bring shapes to the front of the page's object list.
   *
   * @example
   * ```ts
   * editor.bringToFront(['id1', 'id2'])
   * editor.bringToFront([box1, box2])
   * ```
   *
   * @param shapes - The shapes (or shape ids) to move.
   *
   * @public
   */
  bringToFront(shapes) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    const changes = getReorderingShapesChanges(this, "toFront", ids);
    if (changes) this.updateShapes(changes);
    return this;
  }
  /**
   * Flip shape positions.
   *
   * @example
   * ```ts
   * editor.flipShapes([box1, box2], 'horizontal', 32)
   * editor.flipShapes(editor.getSelectedShapeIds(), 'horizontal', 32)
   * ```
   *
   * @param shapes - The ids of the shapes to flip.
   * @param operation - Whether to flip horizontally or vertically.
   *
   * @public
   */
  flipShapes(shapes, operation) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (this.getInstanceState().isReadonly) return this;
    let shapesToFlip = compact(ids.map((id) => this.getShape(id)));
    if (!shapesToFlip.length) return this;
    shapesToFlip = compact(
      shapesToFlip.map((shape) => {
        if (this.isShapeOfType(shape, "group")) {
          return this.getSortedChildIdsForParent(shape.id).map((id) => this.getShape(id));
        }
        return shape;
      }).flat()
    );
    const scaleOriginPage = Box.Common(
      compact(shapesToFlip.map((id) => this.getShapePageBounds(id)))
    ).center;
    this.run(() => {
      for (const shape of shapesToFlip) {
        const bounds = this.getShapeGeometry(shape).bounds;
        const initialPageTransform = this.getShapePageTransform(shape.id);
        if (!initialPageTransform) continue;
        this.resizeShape(
          shape.id,
          { x: operation === "horizontal" ? -1 : 1, y: operation === "vertical" ? -1 : 1 },
          {
            initialBounds: bounds,
            initialPageTransform,
            initialShape: shape,
            mode: "scale_shape",
            isAspectRatioLocked: this.getShapeUtil(shape).isAspectRatioLocked(shape),
            scaleOrigin: scaleOriginPage,
            scaleAxisRotation: 0
          }
        );
      }
    });
    return this;
  }
  /**
   * Stack shape.
   *
   * @example
   * ```ts
   * editor.stackShapes([box1, box2], 'horizontal', 32)
   * editor.stackShapes(editor.getSelectedShapeIds(), 'horizontal', 32)
   * ```
   *
   * @param shapes - The shapes (or shape ids) to stack.
   * @param operation - Whether to stack horizontally or vertically.
   * @param gap - The gap to leave between shapes.
   *
   * @public
   */
  stackShapes(shapes, operation, gap) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (this.getInstanceState().isReadonly) return this;
    const shapesToStack = ids.map((id) => this.getShape(id)).filter((shape) => {
      if (!shape) return false;
      return this.getShapeUtil(shape).canBeLaidOut(shape);
    });
    const len = shapesToStack.length;
    if (gap === 0 && len < 3 || len < 2) return this;
    const pageBounds = Object.fromEntries(
      shapesToStack.map((shape) => [shape.id, this.getShapePageBounds(shape)])
    );
    let val;
    let min;
    let max;
    let dim;
    if (operation === "horizontal") {
      val = "x";
      min = "minX";
      max = "maxX";
      dim = "width";
    } else {
      val = "y";
      min = "minY";
      max = "maxY";
      dim = "height";
    }
    let shapeGap;
    if (gap === 0) {
      const gaps = [];
      shapesToStack.sort((a, b) => pageBounds[a.id][min] - pageBounds[b.id][min]);
      for (let i = 0; i < len - 1; i++) {
        const shape = shapesToStack[i];
        const nextShape = shapesToStack[i + 1];
        const bounds = pageBounds[shape.id];
        const nextBounds = pageBounds[nextShape.id];
        const gap2 = nextBounds[min] - bounds[max];
        const current = gaps.find((g) => g.gap === gap2);
        if (current) {
          current.count++;
        } else {
          gaps.push({ gap: gap2, count: 1 });
        }
      }
      let maxCount = 0;
      gaps.forEach((g) => {
        if (g.count > maxCount) {
          maxCount = g.count;
          shapeGap = g.gap;
        }
      });
      if (maxCount === 1) {
        shapeGap = Math.max(0, gaps.reduce((a, c) => a + c.gap * c.count, 0) / (len - 1));
      }
    } else {
      shapeGap = gap;
    }
    const changes = [];
    let v = pageBounds[shapesToStack[0].id][max];
    shapesToStack.forEach((shape, i) => {
      if (i === 0) return;
      const delta = { x: 0, y: 0 };
      delta[val] = v + shapeGap - pageBounds[shape.id][val];
      const parent = this.getShapeParent(shape);
      const localDelta = parent ? Vec.Rot(delta, -this.getShapePageTransform(parent).decompose().rotation) : delta;
      const translateStartChanges = this.getShapeUtil(shape).onTranslateStart?.(shape);
      changes.push(
        translateStartChanges ? {
          ...translateStartChanges,
          [val]: shape[val] + localDelta[val]
        } : {
          id: shape.id,
          type: shape.type,
          [val]: shape[val] + localDelta[val]
        }
      );
      v += pageBounds[shape.id][dim] + shapeGap;
    });
    this.updateShapes(changes);
    return this;
  }
  /**
   * Pack shapes into a grid centered on their current position. Based on potpack (https://github.com/mapbox/potpack).
   *
   * @example
   * ```ts
   * editor.packShapes([box1, box2], 32)
   * editor.packShapes(editor.getSelectedShapeIds(), 32)
   * ```
   *
   *
   * @param shapes - The shapes (or shape ids) to pack.
   * @param gap - The padding to apply to the packed shapes. Defaults to 16.
   */
  packShapes(shapes, gap) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (this.getInstanceState().isReadonly) return this;
    if (ids.length < 2) return this;
    const shapesToPack = ids.map((id) => this.getShape(id)).filter((shape2) => {
      if (!shape2) return false;
      return this.getShapeUtil(shape2).canBeLaidOut(shape2);
    });
    const shapePageBounds = {};
    const nextShapePageBounds = {};
    let shape, bounds, area = 0;
    for (let i = 0; i < shapesToPack.length; i++) {
      shape = shapesToPack[i];
      bounds = this.getShapePageBounds(shape);
      shapePageBounds[shape.id] = bounds;
      nextShapePageBounds[shape.id] = bounds.clone();
      area += bounds.width * bounds.height;
    }
    const commonBounds = Box.Common(compact(Object.values(shapePageBounds)));
    const maxWidth = commonBounds.width;
    shapesToPack.sort((a, b) => shapePageBounds[b.id].height - shapePageBounds[a.id].height);
    const startWidth = Math.max(Math.ceil(Math.sqrt(area / 0.95)), maxWidth);
    const spaces = [new Box(commonBounds.x, commonBounds.y, startWidth, Infinity)];
    let width = 0;
    let height = 0;
    let space;
    let last2;
    for (let i = 0; i < shapesToPack.length; i++) {
      shape = shapesToPack[i];
      bounds = nextShapePageBounds[shape.id];
      for (let i2 = spaces.length - 1; i2 >= 0; i2--) {
        space = spaces[i2];
        if (bounds.width > space.width || bounds.height > space.height) continue;
        bounds.x = space.x;
        bounds.y = space.y;
        height = Math.max(height, bounds.maxY);
        width = Math.max(width, bounds.maxX);
        if (bounds.width === space.width && bounds.height === space.height) {
          last2 = spaces.pop();
          if (i2 < spaces.length) spaces[i2] = last2;
        } else if (bounds.height === space.height) {
          space.x += bounds.width + gap;
          space.width -= bounds.width + gap;
        } else if (bounds.width === space.width) {
          space.y += bounds.height + gap;
          space.height -= bounds.height + gap;
        } else {
          spaces.push(
            new Box(
              space.x + (bounds.width + gap),
              space.y,
              space.width - (bounds.width + gap),
              bounds.height
            )
          );
          space.y += bounds.height + gap;
          space.height -= bounds.height + gap;
        }
        break;
      }
    }
    const commonAfter = Box.Common(Object.values(nextShapePageBounds));
    const centerDelta = Vec.Sub(commonBounds.center, commonAfter.center);
    let nextBounds;
    const changes = [];
    for (let i = 0; i < shapesToPack.length; i++) {
      shape = shapesToPack[i];
      bounds = shapePageBounds[shape.id];
      nextBounds = nextShapePageBounds[shape.id];
      const delta = Vec.Sub(nextBounds.point, bounds.point).add(centerDelta);
      const parentTransform = this.getShapeParentTransform(shape);
      if (parentTransform) delta.rot(-parentTransform.rotation());
      const change = {
        id: shape.id,
        type: shape.type,
        x: shape.x + delta.x,
        y: shape.y + delta.y
      };
      const translateStartChange = this.getShapeUtil(shape).onTranslateStart?.({
        ...shape,
        ...change
      });
      if (translateStartChange) {
        changes.push({ ...change, ...translateStartChange });
      } else {
        changes.push(change);
      }
    }
    if (changes.length) {
      this.updateShapes(changes);
    }
    return this;
  }
  /**
   * Align shape positions.
   *
   * @example
   * ```ts
   * editor.alignShapes([box1, box2], 'left')
   * editor.alignShapes(editor.getSelectedShapeIds(), 'left')
   * ```
   *
   * @param shapes - The shapes (or shape ids) to align.
   * @param operation - The align operation to apply.
   *
   * @public
   */
  alignShapes(shapes, operation) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (this.getInstanceState().isReadonly) return this;
    if (ids.length < 2) return this;
    const shapesToAlign = compact(ids.map((id) => this.getShape(id)));
    const shapePageBounds = Object.fromEntries(
      shapesToAlign.map((shape) => [shape.id, this.getShapePageBounds(shape)])
    );
    const commonBounds = Box.Common(compact(Object.values(shapePageBounds)));
    const changes = [];
    shapesToAlign.forEach((shape) => {
      const pageBounds = shapePageBounds[shape.id];
      if (!pageBounds) return;
      const delta = { x: 0, y: 0 };
      switch (operation) {
        case "top": {
          delta.y = commonBounds.minY - pageBounds.minY;
          break;
        }
        case "center-vertical": {
          delta.y = commonBounds.midY - pageBounds.minY - pageBounds.height / 2;
          break;
        }
        case "bottom": {
          delta.y = commonBounds.maxY - pageBounds.minY - pageBounds.height;
          break;
        }
        case "left": {
          delta.x = commonBounds.minX - pageBounds.minX;
          break;
        }
        case "center-horizontal": {
          delta.x = commonBounds.midX - pageBounds.minX - pageBounds.width / 2;
          break;
        }
        case "right": {
          delta.x = commonBounds.maxX - pageBounds.minX - pageBounds.width;
          break;
        }
      }
      const parent = this.getShapeParent(shape);
      const localDelta = parent ? Vec.Rot(delta, -this.getShapePageTransform(parent).decompose().rotation) : delta;
      changes.push(this.getChangesToTranslateShape(shape, Vec.Add(shape, localDelta)));
    });
    this.updateShapes(changes);
    return this;
  }
  /**
   * Distribute shape positions.
   *
   * @example
   * ```ts
   * editor.distributeShapes([box1, box2], 'horizontal')
   * editor.distributeShapes(editor.getSelectedShapeIds(), 'horizontal')
   * ```
   *
   * @param shapes - The shapes (or shape ids) to distribute.
   * @param operation - Whether to distribute shapes horizontally or vertically.
   *
   * @public
   */
  distributeShapes(shapes, operation) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (this.getInstanceState().isReadonly) return this;
    if (ids.length < 3) return this;
    const len = ids.length;
    const shapesToDistribute = compact(ids.map((id) => this.getShape(id)));
    const pageBounds = Object.fromEntries(
      shapesToDistribute.map((shape) => [shape.id, this.getShapePageBounds(shape)])
    );
    let val;
    let min;
    let max;
    let mid;
    let dim;
    if (operation === "horizontal") {
      val = "x";
      min = "minX";
      max = "maxX";
      mid = "midX";
      dim = "width";
    } else {
      val = "y";
      min = "minY";
      max = "maxY";
      mid = "midY";
      dim = "height";
    }
    const changes = [];
    const first = shapesToDistribute.sort(
      (a, b) => pageBounds[a.id][min] - pageBounds[b.id][min]
    )[0];
    const last2 = shapesToDistribute.sort((a, b) => pageBounds[b.id][max] - pageBounds[a.id][max])[0];
    const midFirst = pageBounds[first.id][mid];
    const step = (pageBounds[last2.id][mid] - midFirst) / (len - 1);
    const v = midFirst + step;
    shapesToDistribute.filter((shape) => shape !== first && shape !== last2).sort((a, b) => pageBounds[a.id][mid] - pageBounds[b.id][mid]).forEach((shape, i) => {
      const delta = { x: 0, y: 0 };
      delta[val] = v + step * i - pageBounds[shape.id][dim] / 2 - pageBounds[shape.id][val];
      const parent = this.getShapeParent(shape);
      const localDelta = parent ? Vec.Rot(delta, -this.getShapePageTransform(parent).rotation()) : delta;
      changes.push(this.getChangesToTranslateShape(shape, Vec.Add(shape, localDelta)));
    });
    this.updateShapes(changes);
    return this;
  }
  /**
   * Stretch shape sizes and positions to fill their common bounding box.
   *
   * @example
   * ```ts
   * editor.stretchShapes([box1, box2], 'horizontal')
   * editor.stretchShapes(editor.getSelectedShapeIds(), 'horizontal')
   * ```
   *
   * @param shapes - The shapes (or shape ids) to stretch.
   * @param operation - Whether to stretch shapes horizontally or vertically.
   *
   * @public
   */
  stretchShapes(shapes, operation) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (this.getInstanceState().isReadonly) return this;
    if (ids.length < 2) return this;
    const shapesToStretch = compact(ids.map((id) => this.getShape(id)));
    const shapeBounds = Object.fromEntries(ids.map((id) => [id, this.getShapeGeometry(id).bounds]));
    const shapePageBounds = Object.fromEntries(ids.map((id) => [id, this.getShapePageBounds(id)]));
    const commonBounds = Box.Common(compact(Object.values(shapePageBounds)));
    switch (operation) {
      case "vertical": {
        this.run(() => {
          for (const shape of shapesToStretch) {
            const pageRotation = this.getShapePageTransform(shape).rotation();
            if (pageRotation % PI2) continue;
            const bounds = shapeBounds[shape.id];
            const pageBounds = shapePageBounds[shape.id];
            const localOffset = new Vec(0, commonBounds.minY - pageBounds.minY);
            const parentTransform = this.getShapeParentTransform(shape);
            if (parentTransform) localOffset.rot(-parentTransform.rotation());
            const { x, y } = Vec.Add(localOffset, shape);
            this.updateShapes([{ id: shape.id, type: shape.type, x, y }]);
            const scale = new Vec(1, commonBounds.height / pageBounds.height);
            this.resizeShape(shape.id, scale, {
              initialBounds: bounds,
              scaleOrigin: new Vec(pageBounds.center.x, commonBounds.minY),
              isAspectRatioLocked: this.getShapeUtil(shape).isAspectRatioLocked(shape),
              scaleAxisRotation: 0
            });
          }
        });
        break;
      }
      case "horizontal": {
        this.run(() => {
          for (const shape of shapesToStretch) {
            const bounds = shapeBounds[shape.id];
            const pageBounds = shapePageBounds[shape.id];
            const pageRotation = this.getShapePageTransform(shape).rotation();
            if (pageRotation % PI2) continue;
            const localOffset = new Vec(commonBounds.minX - pageBounds.minX, 0);
            const parentTransform = this.getShapeParentTransform(shape);
            if (parentTransform) localOffset.rot(-parentTransform.rotation());
            const { x, y } = Vec.Add(localOffset, shape);
            this.updateShapes([{ id: shape.id, type: shape.type, x, y }]);
            const scale = new Vec(commonBounds.width / pageBounds.width, 1);
            this.resizeShape(shape.id, scale, {
              initialBounds: bounds,
              scaleOrigin: new Vec(commonBounds.minX, pageBounds.center.y),
              isAspectRatioLocked: this.getShapeUtil(shape).isAspectRatioLocked(shape),
              scaleAxisRotation: 0
            });
          }
        });
        break;
      }
    }
    return this;
  }
  /**
   * Resize a shape.
   *
   * @param id - The id of the shape to resize.
   * @param scale - The scale factor to apply to the shape.
   * @param options - Additional options.
   *
   * @public
   */
  resizeShape(shape, scale, options = {}) {
    const id = typeof shape === "string" ? shape : shape.id;
    if (this.getInstanceState().isReadonly) return this;
    if (!Number.isFinite(scale.x)) scale = new Vec(1, scale.y);
    if (!Number.isFinite(scale.y)) scale = new Vec(scale.x, 1);
    const initialShape = options.initialShape ?? this.getShape(id);
    if (!initialShape) return this;
    const scaleOrigin = options.scaleOrigin ?? this.getShapePageBounds(id)?.center;
    if (!scaleOrigin) return this;
    const pageTransform = options.initialPageTransform ? Mat.Cast(options.initialPageTransform) : this.getShapePageTransform(id);
    if (!pageTransform) return this;
    const pageRotation = pageTransform.rotation();
    if (pageRotation == null) return this;
    const scaleAxisRotation = options.scaleAxisRotation ?? pageRotation;
    const initialBounds = options.initialBounds ?? this.getShapeGeometry(id).bounds;
    if (!initialBounds) return this;
    const isAspectRatioLocked = options.isAspectRatioLocked ?? this.getShapeUtil(initialShape).isAspectRatioLocked(initialShape);
    if (!areAnglesCompatible(pageRotation, scaleAxisRotation)) {
      return this._resizeUnalignedShape(id, scale, {
        ...options,
        initialBounds,
        scaleOrigin,
        scaleAxisRotation,
        initialPageTransform: pageTransform,
        isAspectRatioLocked,
        initialShape
      });
    }
    const util = this.getShapeUtil(initialShape);
    if (isAspectRatioLocked) {
      if (Math.abs(scale.x) > Math.abs(scale.y)) {
        scale = new Vec(scale.x, Math.sign(scale.y) * Math.abs(scale.x));
      } else {
        scale = new Vec(Math.sign(scale.x) * Math.abs(scale.y), scale.y);
      }
    }
    if (util.onResize && util.canResize(initialShape)) {
      const newPagePoint = this._scalePagePoint(
        Mat.applyToPoint(pageTransform, new Vec(0, 0)),
        scaleOrigin,
        scale,
        scaleAxisRotation
      );
      const newLocalPoint = this.getPointInParentSpace(initialShape.id, newPagePoint);
      const myScale = new Vec(scale.x, scale.y);
      const areWidthAndHeightAlignedWithCorrectAxis = approximately(
        (pageRotation - scaleAxisRotation) % Math.PI,
        0
      );
      myScale.x = areWidthAndHeightAlignedWithCorrectAxis ? scale.x : scale.y;
      myScale.y = areWidthAndHeightAlignedWithCorrectAxis ? scale.y : scale.x;
      const initialPagePoint = Mat.applyToPoint(pageTransform, new Vec());
      const { x, y } = this.getPointInParentSpace(initialShape.id, initialPagePoint);
      let workingShape = initialShape;
      if (!options.skipStartAndEndCallbacks) {
        workingShape = applyPartialToRecordWithProps(
          initialShape,
          util.onResizeStart?.(initialShape) ?? void 0
        );
      }
      workingShape = applyPartialToRecordWithProps(workingShape, {
        id,
        type: initialShape.type,
        x: newLocalPoint.x,
        y: newLocalPoint.y,
        ...util.onResize(
          { ...initialShape, x, y },
          {
            newPoint: newLocalPoint,
            handle: options.dragHandle ?? "bottom_right",
            // don't set isSingle to true for children
            mode: options.mode ?? "scale_shape",
            scaleX: myScale.x,
            scaleY: myScale.y,
            initialBounds,
            initialShape
          }
        )
      });
      if (!options.skipStartAndEndCallbacks) {
        workingShape = applyPartialToRecordWithProps(
          workingShape,
          util.onResizeEnd?.(initialShape, workingShape) ?? void 0
        );
      }
      this.updateShapes([workingShape]);
    } else {
      const initialPageCenter = Mat.applyToPoint(pageTransform, initialBounds.center);
      const newPageCenter = this._scalePagePoint(
        initialPageCenter,
        scaleOrigin,
        scale,
        scaleAxisRotation
      );
      const initialPageCenterInParentSpace = this.getPointInParentSpace(
        initialShape.id,
        initialPageCenter
      );
      const newPageCenterInParentSpace = this.getPointInParentSpace(initialShape.id, newPageCenter);
      const delta = Vec.Sub(newPageCenterInParentSpace, initialPageCenterInParentSpace);
      this.updateShapes([
        {
          id,
          type: initialShape.type,
          x: initialShape.x + delta.x,
          y: initialShape.y + delta.y
        }
      ]);
    }
    return this;
  }
  /** @internal */
  _scalePagePoint(point, scaleOrigin, scale, scaleAxisRotation) {
    const relativePoint = Vec.RotWith(point, scaleOrigin, -scaleAxisRotation).sub(scaleOrigin);
    const newRelativePagePoint = Vec.MulV(relativePoint, scale);
    const destination = Vec.Add(newRelativePagePoint, scaleOrigin).rotWith(
      scaleOrigin,
      scaleAxisRotation
    );
    return destination;
  }
  /** @internal */
  _resizeUnalignedShape(id, scale, options) {
    const { type } = options.initialShape;
    const shapeScale = new Vec(scale.x, scale.y);
    if (Math.abs(scale.x) > Math.abs(scale.y)) {
      shapeScale.x = Math.sign(scale.x) * Math.abs(scale.y);
    } else {
      shapeScale.y = Math.sign(scale.y) * Math.abs(scale.x);
    }
    this.resizeShape(id, shapeScale, {
      initialShape: options.initialShape,
      initialBounds: options.initialBounds,
      isAspectRatioLocked: options.isAspectRatioLocked
    });
    if (Math.sign(scale.x) * Math.sign(scale.y) < 0) {
      let { rotation } = Mat.Decompose(options.initialPageTransform);
      rotation -= 2 * rotation;
      this.updateShapes([{ id, type, rotation }]);
    }
    const preScaleShapePageCenter = Mat.applyToPoint(
      options.initialPageTransform,
      options.initialBounds.center
    );
    const postScaleShapePageCenter = this._scalePagePoint(
      preScaleShapePageCenter,
      options.scaleOrigin,
      scale,
      options.scaleAxisRotation
    );
    const pageBounds = this.getShapePageBounds(id);
    const pageTransform = this.getShapePageTransform(id);
    const currentPageCenter = pageBounds.center;
    const shapePageTransformOrigin = pageTransform.point();
    if (!currentPageCenter || !shapePageTransformOrigin) return this;
    const pageDelta = Vec.Sub(postScaleShapePageCenter, currentPageCenter);
    const postScaleShapePagePoint = Vec.Add(shapePageTransformOrigin, pageDelta);
    const { x, y } = this.getPointInParentSpace(id, postScaleShapePagePoint);
    this.updateShapes([{ id, type, x, y }]);
    return this;
  }
  /**
   * Get the initial meta value for a shape.
   *
   * @example
   * ```ts
   * editor.getInitialMetaForShape = (shape) => {
   *   if (shape.type === 'note') {
   *     return { createdBy: myCurrentUser.id }
   *   }
   * }
   * ```
   *
   * @param shape - The shape to get the initial meta for.
   *
   * @public
   */
  getInitialMetaForShape(_shape) {
    return {};
  }
  /**
   * Create a single shape.
   *
   * @example
   * ```ts
   * editor.createShape(myShape)
   * editor.createShape({ id: 'box1', type: 'text', props: { text: "ok" } })
   * ```
   *
   * @param shape - The shape (or shape partial) to create.
   *
   * @public
   */
  createShape(shape) {
    this.createShapes([shape]);
    return this;
  }
  /**
   * Create shapes.
   *
   * @example
   * ```ts
   * editor.createShapes([myShape])
   * editor.createShapes([{ id: 'box1', type: 'text', props: { text: "ok" } }])
   * ```
   *
   * @param shapes - The shapes (or shape partials) to create.
   * @param select - Whether to select the created shapes. Defaults to false.
   *
   * @public
   */
  createShapes(shapes) {
    if (!Array.isArray(shapes)) {
      throw Error("Editor.createShapes: must provide an array of shapes or shape partials");
    }
    if (this.getInstanceState().isReadonly) return this;
    if (shapes.length <= 0) return this;
    const currentPageShapeIds = this.getCurrentPageShapeIds();
    const maxShapesReached = shapes.length + currentPageShapeIds.size > this.options.maxShapesPerPage;
    if (maxShapesReached) {
      alertMaxShapes(this);
      return this;
    }
    const focusedGroupId = this.getFocusedGroupId();
    this.run(() => {
      const currentPageShapesSorted = this.getCurrentPageShapesSorted();
      const partials = shapes.map((partial) => {
        if (!partial.id) {
          partial = { id: createShapeId(), ...partial };
        }
        if (!partial.parentId || !(this.store.has(partial.parentId) || shapes.some((p) => p.id === partial.parentId))) {
          let parentId = this.getFocusedGroupId();
          for (let i = currentPageShapesSorted.length - 1; i >= 0; i--) {
            const parent = currentPageShapesSorted[i];
            if (
              // parent.type === 'frame'
              this.getShapeUtil(parent).canReceiveNewChildrenOfType(parent, partial.type) && this.isPointInShape(
                parent,
                // If no parent is provided, then we can treat the
                // shape's provided x/y as being in the page's space.
                { x: partial.x ?? 0, y: partial.y ?? 0 },
                {
                  margin: 0,
                  hitInside: true
                }
              )
            ) {
              parentId = parent.id;
              break;
            }
          }
          const prevParentId = partial.parentId;
          if (parentId === partial.id) {
            parentId = focusedGroupId;
          }
          if (parentId !== prevParentId) {
            partial = { ...partial };
            partial.parentId = parentId;
            if (isShapeId(parentId)) {
              const point = this.getPointInShapeSpace(this.getShape(parentId), {
                x: partial.x ?? 0,
                y: partial.y ?? 0
              });
              partial.x = point.x;
              partial.y = point.y;
              partial.rotation = -this.getShapePageTransform(parentId).rotation() + (partial.rotation ?? 0);
            }
          }
        }
        return partial;
      });
      const parentIndices = /* @__PURE__ */ new Map();
      const shapeRecordsToCreate = [];
      const { opacityForNextShape } = this.getInstanceState();
      for (const partial of partials) {
        const util = this.getShapeUtil(partial);
        let index = partial.index;
        if (!index) {
          const parentId = partial.parentId ?? focusedGroupId;
          if (!parentIndices.has(parentId)) {
            parentIndices.set(parentId, this.getHighestIndexForParent(parentId));
          }
          index = parentIndices.get(parentId);
          parentIndices.set(parentId, getIndexAbove(index));
        }
        const initialProps = util.getDefaultProps();
        for (const [style, propKey] of this.styleProps[partial.type]) {
          ;
          initialProps[propKey] = this.getStyleForNextShape(style);
        }
        let shapeRecordToCreate = this.store.schema.types.shape.create({
          ...partial,
          index,
          opacity: partial.opacity ?? opacityForNextShape,
          parentId: partial.parentId ?? focusedGroupId,
          props: "props" in partial ? { ...initialProps, ...partial.props } : initialProps
        });
        if (shapeRecordToCreate.index === void 0) {
          throw Error("no index!");
        }
        const next = this.getShapeUtil(shapeRecordToCreate).onBeforeCreate?.(shapeRecordToCreate);
        if (next) {
          shapeRecordToCreate = next;
        }
        shapeRecordsToCreate.push(shapeRecordToCreate);
      }
      shapeRecordsToCreate.forEach((shape) => {
        shape.meta = {
          ...this.getInitialMetaForShape(shape),
          ...shape.meta
        };
      });
      this.store.put(shapeRecordsToCreate);
    });
    return this;
  }
  animatingShapes = /* @__PURE__ */ new Map();
  /**
   * Animate a shape.
   *
   * @example
   * ```ts
   * editor.animateShape({ id: 'box1', type: 'box', x: 100, y: 100 })
   * editor.animateShape({ id: 'box1', type: 'box', x: 100, y: 100 }, { animation: { duration: 100, ease: t => t*t } })
   * ```
   *
   * @param partial - The shape partial to update.
   * @param options - The animation's options.
   *
   * @public
   */
  animateShape(partial, opts = { animation: DEFAULT_ANIMATION_OPTIONS }) {
    return this.animateShapes([partial], opts);
  }
  /**
   * Animate shapes.
   *
   * @example
   * ```ts
   * editor.animateShapes([{ id: 'box1', type: 'box', x: 100, y: 100 }])
   * editor.animateShapes([{ id: 'box1', type: 'box', x: 100, y: 100 }], { animation: { duration: 100, ease: t => t*t } })
   * ```
   *
   * @param partials - The shape partials to update.
   * @param options - The animation's options.
   *
   * @public
   */
  animateShapes(partials, opts = { animation: DEFAULT_ANIMATION_OPTIONS }) {
    if (!opts.animation) return this;
    const { duration = 500, easing = EASINGS.linear } = opts.animation;
    const animationId = uniqueId();
    let remaining = duration;
    let t;
    const animations = [];
    let partial, result;
    for (let i = 0, n = partials.length; i < n; i++) {
      partial = partials[i];
      if (!partial) continue;
      const shape = this.getShape(partial.id);
      if (!shape) continue;
      result = {
        start: structuredClone(shape),
        end: applyPartialToRecordWithProps(structuredClone(shape), partial)
      };
      animations.push(result);
      this.animatingShapes.set(shape.id, animationId);
    }
    const handleTick = (elapsed) => {
      remaining -= elapsed;
      if (remaining < 0) {
        const { animatingShapes: animatingShapes2 } = this;
        const partialsToUpdate = partials.filter(
          (p) => p && animatingShapes2.get(p.id) === animationId
        );
        if (partialsToUpdate.length) {
          this.updateShapes(partialsToUpdate);
        }
        this.off("tick", handleTick);
        return;
      }
      t = easing(1 - remaining / duration);
      const { animatingShapes } = this;
      const updates = [];
      let animationIdForShape;
      for (let i = 0, n = animations.length; i < n; i++) {
        const { start, end } = animations[i];
        animationIdForShape = animatingShapes.get(start.id);
        if (animationIdForShape !== animationId) continue;
        updates.push({
          ...end,
          x: start.x + (end.x - start.x) * t,
          y: start.y + (end.y - start.y) * t,
          opacity: start.opacity + (end.opacity - start.opacity) * t,
          rotation: start.rotation + (end.rotation - start.rotation) * t,
          props: this.getShapeUtil(end).getInterpolatedProps?.(start, end, t) ?? end.props
        });
      }
      this._updateShapes(updates);
    };
    this.on("tick", handleTick);
    return this;
  }
  groupShapes(shapes, options = {}) {
    const { groupId = createShapeId(), select = true } = options;
    if (!Array.isArray(shapes)) {
      throw Error("Editor.groupShapes: must provide an array of shapes or shape ids");
    }
    if (this.getInstanceState().isReadonly) return this;
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (ids.length <= 1) return this;
    const shapesToGroup = compact(
      (this._shouldIgnoreShapeLock ? ids : this._getUnlockedShapeIds(ids)).map(
        (id) => this.getShape(id)
      )
    );
    const sortedShapeIds = shapesToGroup.sort(sortByIndex).map((s) => s.id);
    const pageBounds = Box.Common(compact(shapesToGroup.map((id) => this.getShapePageBounds(id))));
    const { x, y } = pageBounds.point;
    const parentId = this.findCommonAncestor(shapesToGroup) ?? this.getCurrentPageId();
    if (this.getCurrentToolId() !== "select") return this;
    if (!this.isIn("select.idle")) {
      this.cancel();
    }
    const shapesWithRootParent = shapesToGroup.filter((shape) => shape.parentId === parentId).sort(sortByIndex);
    const highestIndex = shapesWithRootParent[shapesWithRootParent.length - 1]?.index;
    this.run(() => {
      this.createShapes([
        {
          id: groupId,
          type: "group",
          parentId,
          index: highestIndex,
          x,
          y,
          opacity: 1,
          props: {}
        }
      ]);
      this.reparentShapes(sortedShapeIds, groupId);
      if (select) {
        this.select(groupId);
      }
    });
    return this;
  }
  ungroupShapes(shapes, options = {}) {
    if (this.getInstanceState().isReadonly) return this;
    const { select = true } = options;
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    const shapesToUngroup = compact(
      (this._shouldIgnoreShapeLock ? ids : this._getUnlockedShapeIds(ids)).map(
        (id) => this.getShape(id)
      )
    );
    if (shapesToUngroup.length === 0) return this;
    if (this.getCurrentToolId() !== "select") return this;
    if (!this.isIn("select.idle")) {
      this.cancel();
    }
    const idsToSelect = /* @__PURE__ */ new Set();
    const groups = [];
    shapesToUngroup.forEach((shape) => {
      if (this.isShapeOfType(shape, "group")) {
        groups.push(shape);
      } else {
        idsToSelect.add(shape.id);
      }
    });
    if (groups.length === 0) return this;
    this.run(() => {
      let group;
      for (let i = 0, n = groups.length; i < n; i++) {
        group = groups[i];
        const childIds = this.getSortedChildIdsForParent(group.id);
        for (let j = 0, n2 = childIds.length; j < n2; j++) {
          idsToSelect.add(childIds[j]);
        }
        this.reparentShapes(childIds, group.parentId, group.index);
      }
      this.deleteShapes(groups.map((group2) => group2.id));
      if (select) {
        this.select(...idsToSelect);
      }
    });
    return this;
  }
  /**
   * Update a shape using a partial of the shape.
   *
   * @example
   * ```ts
   * editor.updateShape({ id: 'box1', type: 'geo', props: { w: 100, h: 100 } })
   * ```
   *
   * @param partial - The shape partial to update.
   *
   * @public
   */
  updateShape(partial) {
    this.updateShapes([partial]);
    return this;
  }
  /**
   * Update shapes using partials of each shape.
   *
   * @example
   * ```ts
   * editor.updateShapes([{ id: 'box1', type: 'geo', props: { w: 100, h: 100 } }])
   * ```
   *
   * @param partials - The shape partials to update.
   *
   * @public
   */
  updateShapes(partials) {
    const compactedPartials = Array(partials.length);
    for (let i = 0, n = partials.length; i < n; i++) {
      const partial = partials[i];
      if (!partial) continue;
      const shape = this.getShape(partial.id);
      if (!shape) continue;
      if (!this._shouldIgnoreShapeLock) {
        if (shape.isLocked) {
          if (!(Object.hasOwn(partial, "isLocked") && !partial.isLocked)) {
            continue;
          }
        } else if (this.isShapeOrAncestorLocked(shape)) {
          continue;
        }
      }
      this.animatingShapes.delete(partial.id);
      compactedPartials.push(partial);
    }
    this._updateShapes(compactedPartials);
    return this;
  }
  /** @internal */
  _updateShapes = (_partials) => {
    if (this.getInstanceState().isReadonly) return;
    this.run(() => {
      const updates = [];
      let shape;
      let updated;
      for (let i = 0, n = _partials.length; i < n; i++) {
        const partial = _partials[i];
        if (!partial) continue;
        shape = this.getShape(partial.id);
        if (!shape) continue;
        updated = applyPartialToRecordWithProps(shape, partial);
        if (updated === shape) continue;
        updated = this.getShapeUtil(shape).onBeforeUpdate?.(shape, updated) ?? updated;
        updates.push(updated);
      }
      this.store.put(updates);
    });
  };
  /** @internal */
  _getUnlockedShapeIds(ids) {
    return ids.filter((id) => !this.getShape(id)?.isLocked);
  }
  deleteShapes(_ids) {
    if (this.getInstanceState().isReadonly) return this;
    if (!Array.isArray(_ids)) {
      throw Error("Editor.deleteShapes: must provide an array of shapes or shapeIds");
    }
    const shapeIds = typeof _ids[0] === "string" ? _ids : _ids.map((s) => s.id);
    const shapeIdsToDelete = this._shouldIgnoreShapeLock ? shapeIds : this._getUnlockedShapeIds(shapeIds);
    if (shapeIdsToDelete.length === 0) return this;
    const allShapeIdsToDelete = new Set(shapeIdsToDelete);
    for (const id of shapeIdsToDelete) {
      this.visitDescendants(id, (childId) => {
        allShapeIdsToDelete.add(childId);
      });
    }
    return this.run(() => this.store.remove([...allShapeIdsToDelete]));
  }
  deleteShape(_id) {
    this.deleteShapes([typeof _id === "string" ? _id : _id.id]);
    return this;
  }
  /* --------------------- Styles --------------------- */
  /**
   * Get all the current styles among the users selected shapes
   *
   * @internal
   */
  _extractSharedStyles(shape, sharedStyleMap) {
    if (this.isShapeOfType(shape, "group")) {
      const childIds = this._parentIdsToChildIds.get()[shape.id];
      if (!childIds) return;
      for (let i = 0, n = childIds.length; i < n; i++) {
        this._extractSharedStyles(this.getShape(childIds[i]), sharedStyleMap);
      }
    } else {
      for (const [style, propKey] of this.styleProps[shape.type]) {
        sharedStyleMap.applyValue(style, getOwnProperty(shape.props, propKey));
      }
    }
  }
  _getSelectionSharedStyles() {
    const selectedShapes = this.getSelectedShapes();
    const sharedStyles = new SharedStyleMap();
    for (const selectedShape of selectedShapes) {
      this._extractSharedStyles(selectedShape, sharedStyles);
    }
    return sharedStyles;
  }
  /**
   * Get the style for the next shape.
   *
   * @example
   * ```ts
   * const color = editor.getStyleForNextShape(DefaultColorStyle)
   * ```
   *
   * @param style - The style to get.
   *
   * @public */
  getStyleForNextShape(style) {
    const value = this.getInstanceState().stylesForNextShape[style.id];
    return value === void 0 ? style.defaultValue : value;
  }
  getShapeStyleIfExists(shape, style) {
    const styleKey = this.styleProps[shape.type].get(style);
    if (styleKey === void 0) return void 0;
    return getOwnProperty(shape.props, styleKey);
  }
  getSharedStyles() {
    if (this.isIn("select") && this.getSelectedShapeIds().length > 0) {
      return this._getSelectionSharedStyles();
    }
    const currentTool = this.root.getCurrent();
    const styles = new SharedStyleMap();
    if (!currentTool) return styles;
    if (currentTool.shapeType) {
      for (const style of this.styleProps[currentTool.shapeType].keys()) {
        styles.applyValue(style, this.getStyleForNextShape(style));
      }
    }
    return styles;
  }
  getSharedOpacity() {
    if (this.isIn("select") && this.getSelectedShapeIds().length > 0) {
      const shapesToCheck = [];
      const addShape = (shapeId) => {
        const shape = this.getShape(shapeId);
        if (!shape) return;
        if (this.isShapeOfType(shape, "group")) {
          for (const childId of this.getSortedChildIdsForParent(shape.id)) {
            addShape(childId);
          }
        } else {
          shapesToCheck.push(shape);
        }
      };
      for (const shapeId of this.getSelectedShapeIds()) {
        addShape(shapeId);
      }
      let opacity = null;
      for (const shape of shapesToCheck) {
        if (opacity === null) {
          opacity = shape.opacity;
        } else if (opacity !== shape.opacity) {
          return { type: "mixed" };
        }
      }
      if (opacity !== null) return { type: "shared", value: opacity };
    }
    return { type: "shared", value: this.getInstanceState().opacityForNextShape };
  }
  /**
   * Set the opacity for the next shapes. This will effect subsequently created shapes.
   *
   * @example
   * ```ts
   * editor.setOpacityForNextShapes(0.5)
   * ```
   *
   * @param opacity - The opacity to set. Must be a number between 0 and 1 inclusive.
   * @param historyOptions - The history options for the change.
   */
  setOpacityForNextShapes(opacity, historyOptions) {
    this.updateInstanceState({ opacityForNextShape: opacity }, historyOptions);
    return this;
  }
  /**
   * Set the current opacity. This will effect any selected shapes.
   *
   * @example
   * ```ts
   * editor.setOpacityForSelectedShapes(0.5)
   * ```
   *
   * @param opacity - The opacity to set. Must be a number between 0 and 1 inclusive.
   */
  setOpacityForSelectedShapes(opacity) {
    const selectedShapes = this.getSelectedShapes();
    if (selectedShapes.length > 0) {
      const shapesToUpdate = [];
      const addShapeById = (shape) => {
        if (this.isShapeOfType(shape, "group")) {
          const childIds = this.getSortedChildIdsForParent(shape);
          for (const childId of childIds) {
            addShapeById(this.getShape(childId));
          }
        } else {
          shapesToUpdate.push(shape);
        }
      };
      for (const id of selectedShapes) {
        addShapeById(id);
      }
      this.updateShapes(
        shapesToUpdate.map((shape) => {
          return {
            id: shape.id,
            type: shape.type,
            opacity
          };
        })
      );
    }
    return this;
  }
  /**
   * Set the value of a {@link @tldraw/tlschema#StyleProp} for the next shapes. This change will be applied to subsequently created shapes.
   *
   * @example
   * ```ts
   * editor.setStyleForNextShapes(DefaultColorStyle, 'red')
   * editor.setStyleForNextShapes(DefaultColorStyle, 'red', { ephemeral: true })
   * ```
   *
   * @param style - The style to set.
   * @param value - The value to set.
   * @param historyOptions - The history options for the change.
   *
   * @public
   */
  setStyleForNextShapes(style, value, historyOptions) {
    const stylesForNextShape = this.getInstanceState().stylesForNextShape;
    this.updateInstanceState(
      { stylesForNextShape: { ...stylesForNextShape, [style.id]: value } },
      historyOptions
    );
    return this;
  }
  /**
   * Set the value of a {@link @tldraw/tlschema#StyleProp}. This change will be applied to the currently selected shapes.
   *
   * @example
   * ```ts
   * editor.setStyleForSelectedShapes(DefaultColorStyle, 'red')
   * ```
   *
   * @param style - The style to set.
   * @param value - The value to set.
   * @param historyOptions - The history options for the change.
   *
   * @public
   */
  setStyleForSelectedShapes(style, value) {
    const selectedShapes = this.getSelectedShapes();
    if (selectedShapes.length > 0) {
      const updates = [];
      const addShapeById = (shape) => {
        if (this.isShapeOfType(shape, "group")) {
          const childIds = this.getSortedChildIdsForParent(shape.id);
          for (const childId of childIds) {
            addShapeById(this.getShape(childId));
          }
        } else {
          const util = this.getShapeUtil(shape);
          const stylePropKey = this.styleProps[shape.type].get(style);
          if (stylePropKey) {
            const shapePartial = {
              id: shape.id,
              type: shape.type,
              props: { [stylePropKey]: value }
            };
            updates.push({
              util,
              originalShape: shape,
              updatePartial: shapePartial
            });
          }
        }
      };
      for (const shape of selectedShapes) {
        addShapeById(shape);
      }
      this.updateShapes(updates.map(({ updatePartial }) => updatePartial));
    }
    return this;
  }
  /* --------------------- Content -------------------- */
  /** @internal */
  externalAssetContentHandlers = {
    file: null,
    url: null
  };
  /**
   * Register an external asset handler. This handler will be called when the editor needs to
   * create an asset for some external content, like an image/video file or a bookmark URL. For
   * example, the 'file' type handler will be called when a user drops an image onto the canvas.
   *
   * The handler should extract any relevant metadata for the asset, upload it to blob storage
   * using {@link Editor.uploadAsset} if needed, and return the asset with the metadata & uploaded
   * URL.
   *
   * @example
   * ```ts
   * editor.registerExternalAssetHandler('file', myHandler)
   * ```
   *
   * @param type - The type of external content.
   * @param handler - The handler to use for this content type.
   *
   * @public
   */
  registerExternalAssetHandler(type, handler) {
    this.externalAssetContentHandlers[type] = handler;
    return this;
  }
  /**
   * Get an asset for an external asset content type.
   *
   * @example
   * ```ts
   * const asset = await editor.getAssetForExternalContent({ type: 'file', file: myFile })
   * const asset = await editor.getAssetForExternalContent({ type: 'url', url: myUrl })
   * ```
   *
   * @param info - Info about the external content.
   * @returns The asset.
   */
  async getAssetForExternalContent(info) {
    return await this.externalAssetContentHandlers[info.type]?.(info);
  }
  hasExternalAssetHandler(type) {
    return !!this.externalAssetContentHandlers[type];
  }
  /** @internal */
  externalContentHandlers = {
    text: null,
    files: null,
    embed: null,
    "svg-text": null,
    url: null
  };
  /**
   * Register an external content handler. This handler will be called when the editor receives
   * external content of the provided type. For example, the 'image' type handler will be called
   * when a user drops an image onto the canvas.
   *
   * @example
   * ```ts
   * editor.registerExternalContentHandler('text', myHandler)
   * ```
   *
   * @param type - The type of external content.
   * @param handler - The handler to use for this content type.
   *
   * @public
   */
  registerExternalContentHandler(type, handler) {
    this.externalContentHandlers[type] = handler;
    return this;
  }
  /**
   * Handle external content, such as files, urls, embeds, or plain text which has been put into the app, for example by pasting external text or dropping external images onto canvas.
   *
   * @param info - Info about the external content.
   */
  async putExternalContent(info) {
    return this.externalContentHandlers[info.type]?.(info);
  }
  /**
   * Get content that can be exported for the given shape ids.
   *
   * @param shapes - The shapes (or shape ids) to get content for.
   *
   * @returns The exported content.
   *
   * @public
   */
  getContentFromCurrentPage(shapes) {
    const ids = typeof shapes[0] === "string" ? shapes : shapes.map((s) => s.id);
    if (!ids) return;
    if (ids.length === 0) return;
    const shapeIds = this.getShapeAndDescendantIds(ids);
    return withIsolatedShapes(this, shapeIds, (bindingIdsToKeep) => {
      const bindings = [];
      for (const id of bindingIdsToKeep) {
        const binding = this.getBinding(id);
        if (!binding) continue;
        bindings.push(binding);
      }
      const rootShapeIds = [];
      const shapes2 = [];
      for (const shapeId of shapeIds) {
        const shape = this.getShape(shapeId);
        if (!shape) continue;
        const isRootShape = !shapeIds.has(shape.parentId);
        if (isRootShape) {
          const pageTransform = this.getShapePageTransform(shape.id);
          const pagePoint = pageTransform.point();
          shapes2.push({
            ...shape,
            x: pagePoint.x,
            y: pagePoint.y,
            rotation: pageTransform.rotation(),
            parentId: this.getCurrentPageId()
          });
          rootShapeIds.push(shape.id);
        } else {
          shapes2.push(shape);
        }
      }
      const assets = [];
      const seenAssetIds = /* @__PURE__ */ new Set();
      for (const shape of shapes2) {
        if (!("assetId" in shape.props)) continue;
        const assetId = shape.props.assetId;
        if (!assetId || seenAssetIds.has(assetId)) continue;
        seenAssetIds.add(assetId);
        const asset = this.getAsset(assetId);
        if (!asset) continue;
        assets.push(asset);
      }
      return {
        schema: this.store.schema.serialize(),
        shapes: shapes2,
        rootShapeIds,
        bindings,
        assets
      };
    });
  }
  async resolveAssetsInContent(content) {
    if (!content) return void 0;
    const assets = [];
    await Promise.allSettled(
      content.assets.map(async (asset) => {
        if ((asset.type === "image" || asset.type === "video") && !asset.props.src?.startsWith("data:image") && !asset.props.src?.startsWith("http")) {
          const assetWithDataUrl = structuredClone(asset);
          const objectUrl = await this.store.props.assets.resolve(asset, {
            screenScale: 1,
            steppedScreenScale: 1,
            dpr: 1,
            networkEffectiveType: null,
            shouldResolveToOriginal: true
          });
          assetWithDataUrl.props.src = await FileHelpers.blobToDataUrl(
            await fetch(objectUrl).then((r) => r.blob())
          );
          assets.push(assetWithDataUrl);
        } else {
          assets.push(asset);
        }
      })
    );
    content.assets = assets;
    return content;
  }
  /**
   * Place content into the editor.
   *
   * @param content - The content.
   * @param options - Options for placing the content.
   *
   * @public
   */
  putContentOntoCurrentPage(content, options = {}) {
    if (this.getInstanceState().isReadonly) return this;
    if (!content.schema) {
      throw Error("Could not put content:\ncontent is missing a schema.");
    }
    const { select = false, preserveIds = false, preservePosition = false } = options;
    let { point = void 0 } = options;
    const currentPageId = this.getCurrentPageId();
    const { rootShapeIds } = content;
    const assets = [];
    const shapes = [];
    const bindings = [];
    const store = {
      store: {
        ...Object.fromEntries(content.assets.map((asset) => [asset.id, asset])),
        ...Object.fromEntries(content.shapes.map((shape) => [shape.id, shape])),
        ...Object.fromEntries(
          content.bindings?.map((bindings2) => [bindings2.id, bindings2]) ?? []
        )
      },
      schema: content.schema
    };
    const result = this.store.schema.migrateStoreSnapshot(store);
    if (result.type === "error") {
      throw Error("Could not put content: could not migrate content");
    }
    for (const record of Object.values(result.value)) {
      switch (record.typeName) {
        case "asset": {
          assets.push(record);
          break;
        }
        case "shape": {
          shapes.push(record);
          break;
        }
        case "binding": {
          bindings.push(record);
          break;
        }
      }
    }
    const shapeIdMap = new Map(
      preserveIds ? shapes.map((shape) => [shape.id, shape.id]) : shapes.map((shape) => [shape.id, createShapeId()])
    );
    const bindingIdMap = new Map(
      preserveIds ? bindings.map((binding) => [binding.id, binding.id]) : bindings.map((binding) => [binding.id, createBindingId()])
    );
    let pasteParentId = this.getCurrentPageId();
    let lowestDepth = Infinity;
    let lowestAncestors = [];
    for (const shape of this.getSelectedShapes()) {
      if (lowestDepth === 0) break;
      const isFrame = this.isShapeOfType(shape, "frame");
      const ancestors = this.getShapeAncestors(shape);
      if (isFrame) ancestors.push(shape);
      const depth = isFrame ? ancestors.length + 1 : ancestors.length;
      if (depth < lowestDepth) {
        lowestDepth = depth;
        lowestAncestors = ancestors;
        pasteParentId = isFrame ? shape.id : shape.parentId;
      } else if (depth === lowestDepth) {
        if (lowestAncestors.length !== ancestors.length) {
          throw Error(`Ancestors: ${lowestAncestors.length} !== ${ancestors.length}`);
        }
        if (lowestAncestors.length === 0) {
          pasteParentId = currentPageId;
          break;
        } else {
          pasteParentId = currentPageId;
          for (let i = 0; i < lowestAncestors.length; i++) {
            if (ancestors[i] !== lowestAncestors[i]) break;
            pasteParentId = ancestors[i].id;
          }
        }
      }
    }
    let isDuplicating = false;
    if (!isPageId(pasteParentId)) {
      const parent = this.getShape(pasteParentId);
      if (parent) {
        if (!this.getViewportPageBounds().includes(this.getShapePageBounds(parent))) {
          pasteParentId = currentPageId;
        } else {
          if (rootShapeIds.length === 1) {
            const rootShape = shapes.find((s) => s.id === rootShapeIds[0]);
            if (this.isShapeOfType(parent, "frame") && this.isShapeOfType(rootShape, "frame") && rootShape.props.w === parent?.props.w && rootShape.props.h === parent?.props.h) {
              isDuplicating = true;
            }
          }
        }
      } else {
        pasteParentId = currentPageId;
      }
    }
    if (!isDuplicating) {
      isDuplicating = shapeIdMap.has(pasteParentId);
    }
    if (isDuplicating) {
      pasteParentId = this.getShape(pasteParentId).parentId;
    }
    let index = this.getHighestIndexForParent(pasteParentId);
    const rootShapes = [];
    const newShapes = shapes.map((oldShape) => {
      const newId = shapeIdMap.get(oldShape.id);
      const newShape = { ...oldShape, id: newId };
      if (rootShapeIds.includes(oldShape.id)) {
        newShape.parentId = currentPageId;
        rootShapes.push(newShape);
      }
      if (shapeIdMap.has(newShape.parentId)) {
        newShape.parentId = shapeIdMap.get(oldShape.parentId);
      } else {
        rootShapeIds.push(newShape.id);
        newShape.index = index;
        index = getIndexAbove(index);
      }
      return newShape;
    });
    if (newShapes.length + this.getCurrentPageShapeIds().size > this.options.maxShapesPerPage) {
      alertMaxShapes(this);
      return this;
    }
    const newBindings = bindings.map(
      (oldBinding) => ({
        ...oldBinding,
        id: assertExists(bindingIdMap.get(oldBinding.id)),
        fromId: assertExists(shapeIdMap.get(oldBinding.fromId)),
        toId: assertExists(shapeIdMap.get(oldBinding.toId))
      })
    );
    const assetsToCreate = [];
    const assetsToUpdate = [];
    for (const asset of assets) {
      if (this.store.has(asset.id)) {
        continue;
      }
      if ((asset.type === "image" || asset.type === "video") && asset.props.src?.startsWith("data:image")) {
        assetsToUpdate.push(structuredClone(asset));
        asset.props.src = null;
      }
      assetsToCreate.push(asset);
    }
    Promise.allSettled(
      assetsToUpdate.map(async (asset) => {
        const file = await dataUrlToFile(
          asset.props.src,
          asset.props.name,
          asset.props.mimeType ?? "image/png"
        );
        const newAsset = await this.getAssetForExternalContent({ type: "file", file });
        if (!newAsset) {
          this.deleteAssets([asset.id]);
          return;
        }
        this.updateAssets([{ ...newAsset, id: asset.id }]);
      })
    );
    this.run(() => {
      if (assetsToCreate.length > 0) {
        this.createAssets(assetsToCreate);
      }
      this.createShapes(newShapes);
      this.createBindings(newBindings);
      if (select) {
        this.select(...rootShapes.map((s) => s.id));
      }
      if (pasteParentId !== currentPageId) {
        this.reparentShapes(
          rootShapes.map((s) => s.id),
          pasteParentId
        );
      }
      const newCreatedShapes = newShapes.map((s) => this.getShape(s.id));
      const bounds = Box.Common(newCreatedShapes.map((s) => this.getShapePageBounds(s)));
      if (point === void 0) {
        if (!isPageId(pasteParentId)) {
          const shape = this.getShape(pasteParentId);
          point = Mat.applyToPoint(
            this.getShapePageTransform(shape),
            this.getShapeGeometry(shape).bounds.center
          );
        } else {
          const viewportPageBounds = this.getViewportPageBounds();
          if (preservePosition || viewportPageBounds.includes(Box.From(bounds))) {
            point = bounds.center;
          } else {
            point = viewportPageBounds.center;
          }
        }
      }
      if (rootShapes.length === 1) {
        const onlyRoot = rootShapes[0];
        if (this.isShapeOfType(onlyRoot, "frame")) {
          while (this.getShapesAtPoint(point).some(
            (shape) => this.isShapeOfType(shape, "frame") && shape.props.w === onlyRoot.props.w && shape.props.h === onlyRoot.props.h
          )) {
            point.x += bounds.w + 16;
          }
        }
      }
      const pageCenter = Box.Common(
        compact(rootShapes.map(({ id }) => this.getShapePageBounds(id)))
      ).center;
      const offset = Vec.Sub(point, pageCenter);
      this.updateShapes(
        rootShapes.map(({ id }) => {
          const s = this.getShape(id);
          const localRotation = this.getShapeParentTransform(id).decompose().rotation;
          const localDelta = Vec.Rot(offset, -localRotation);
          return { id: s.id, type: s.type, x: s.x + localDelta.x, y: s.y + localDelta.y };
        })
      );
    });
    return this;
  }
  /**
   * Get an exported SVG element of the given shapes.
   *
   * @param ids - The shapes (or shape ids) to export.
   * @param opts - Options for the export.
   *
   * @returns The SVG element.
   *
   * @public
   */
  async getSvgElement(shapes, opts = {}) {
    const result = await getSvgJsx(this, shapes, opts);
    if (!result) return void 0;
    const fragment = document.createDocumentFragment();
    const root = createRoot(fragment);
    flushSync(() => {
      root.render(result.jsx);
    });
    const svg = fragment.firstElementChild;
    assert(svg instanceof SVGSVGElement, "Expected an SVG element");
    root.unmount();
    return { svg, width: result.width, height: result.height };
  }
  /**
   * Get an exported SVG string of the given shapes.
   *
   * @param ids - The shapes (or shape ids) to export.
   * @param opts - Options for the export.
   *
   * @returns The SVG element.
   *
   * @public
   */
  async getSvgString(shapes, opts = {}) {
    const result = await this.getSvgElement(shapes, opts);
    if (!result) return void 0;
    const serializer = new XMLSerializer();
    return {
      svg: serializer.serializeToString(result.svg),
      width: result.width,
      height: result.height
    };
  }
  /** @deprecated Use {@link Editor.getSvgString} or {@link Editor.getSvgElement} instead. */
  async getSvg(shapes, opts = {}) {
    const result = await this.getSvgElement(shapes, opts);
    if (!result) return void 0;
    return result.svg;
  }
  /* --------------------- Events --------------------- */
  /**
   * The app's current input state.
   *
   * @public
   */
  inputs = {
    /** The most recent pointer down's position in the current page space. */
    originPagePoint: new Vec(),
    /** The most recent pointer down's position in screen space. */
    originScreenPoint: new Vec(),
    /** The previous pointer position in the current page space. */
    previousPagePoint: new Vec(),
    /** The previous pointer position in screen space. */
    previousScreenPoint: new Vec(),
    /** The most recent pointer position in the current page space. */
    currentPagePoint: new Vec(),
    /** The most recent pointer position in screen space. */
    currentScreenPoint: new Vec(),
    /** A set containing the currently pressed keys. */
    keys: /* @__PURE__ */ new Set(),
    /** A set containing the currently pressed buttons. */
    buttons: /* @__PURE__ */ new Set(),
    /** Whether the input is from a pe. */
    isPen: false,
    /** Whether the shift key is currently pressed. */
    shiftKey: false,
    /** Whether the control or command key is currently pressed. */
    ctrlKey: false,
    /** Whether the alt or option key is currently pressed. */
    altKey: false,
    /** Whether the user is dragging. */
    isDragging: false,
    /** Whether the user is pointing. */
    isPointing: false,
    /** Whether the user is pinching. */
    isPinching: false,
    /** Whether the user is editing. */
    isEditing: false,
    /** Whether the user is panning. */
    isPanning: false,
    /** Velocity of mouse pointer, in pixels per millisecond */
    pointerVelocity: new Vec()
  };
  /**
   * Update the input points from a pointer, pinch, or wheel event.
   *
   * @param info - The event info.
   */
  _updateInputsFromEvent(info) {
    const {
      pointerVelocity,
      previousScreenPoint,
      previousPagePoint,
      currentScreenPoint,
      currentPagePoint
    } = this.inputs;
    const { screenBounds } = this.store.unsafeGetWithoutCapture(TLINSTANCE_ID);
    const { x: cx, y: cy, z: cz } = unsafe__withoutCapture(() => this.getCamera());
    const sx = info.point.x - screenBounds.x;
    const sy = info.point.y - screenBounds.y;
    const sz = info.point.z ?? 0.5;
    previousScreenPoint.setTo(currentScreenPoint);
    previousPagePoint.setTo(currentPagePoint);
    currentScreenPoint.set(sx, sy);
    const nx = sx / cz - cx;
    const ny = sy / cz - cy;
    if (isFinite(nx) && isFinite(ny)) {
      currentPagePoint.set(nx, ny, sz);
    }
    this.inputs.isPen = info.type === "pointer" && info.isPen;
    if (info.name === "pointer_down" || this.inputs.isPinching) {
      pointerVelocity.set(0, 0);
      this.inputs.originScreenPoint.setTo(currentScreenPoint);
      this.inputs.originPagePoint.setTo(currentPagePoint);
    }
    this.run(
      () => {
        this.store.put([
          {
            id: TLPOINTER_ID,
            typeName: "pointer",
            x: currentPagePoint.x,
            y: currentPagePoint.y,
            lastActivityTimestamp: (
              // If our pointer moved only because we're following some other user, then don't
              // update our last activity timestamp; otherwise, update it to the current timestamp.
              (info.type === "pointer" && info.pointerId === INTERNAL_POINTER_IDS.CAMERA_MOVE ? this.store.unsafeGetWithoutCapture(TLPOINTER_ID)?.lastActivityTimestamp ?? this._tickManager.now : this._tickManager.now)
            ),
            meta: {}
          }
        ]);
      },
      { history: "ignore" }
    );
  }
  /**
   * Dispatch a cancel event.
   *
   * @example
   * ```ts
   * editor.cancel()
   * ```
   *
   * @public
   */
  cancel() {
    this.dispatch({ type: "misc", name: "cancel" });
    return this;
  }
  /**
   * Dispatch an interrupt event.
   *
   * @example
   * ```ts
   * editor.interrupt()
   * ```
   *
   * @public
   */
  interrupt() {
    this.dispatch({ type: "misc", name: "interrupt" });
    return this;
  }
  /**
   * Dispatch a complete event.
   *
   * @example
   * ```ts
   * editor.complete()
   * ```
   *
   * @public
   */
  complete() {
    this.dispatch({ type: "misc", name: "complete" });
    return this;
  }
  /**
   * Puts the editor into focused mode.
   *
   * This makes the editor eligible to receive keyboard events and some pointer events (move, wheel).
   *
   * @example
   * ```ts
   * editor.focus()
   * ```
   *
   * By default this also dispatches a 'focus' event to the container element. To prevent this, pass `focusContainer: false`.
   *
   * @example
   * ```ts
   * editor.focus({ focusContainer: false })
   * ```
   *
   * @public
   */
  focus({ focusContainer = true } = {}) {
    if (this.getIsFocused()) return this;
    if (focusContainer) this.focusManager.focus();
    this.updateInstanceState({ isFocused: true });
    return this;
  }
  /**
   * Switches off the editor's focused mode.
   *
   * This makes the editor ignore keyboard events and some pointer events (move, wheel).
   *
   * @example
   * ```ts
   * editor.blur()
   * ```
   * By default this also dispatches a 'blur' event to the container element. To prevent this, pass `blurContainer: false`.
   *
   * @example
   * ```ts
   * editor.blur({ blurContainer: false })
   * ```
   *
   * @public
   */
  blur({ blurContainer = true } = {}) {
    if (!this.getIsFocused()) return this;
    if (blurContainer) {
      this.focusManager.blur();
    } else {
      this.complete();
    }
    this.updateInstanceState({ isFocused: false });
    return this;
  }
  getIsFocused() {
    return this.getInstanceState().isFocused;
  }
  /**
   * @public
   * @returns a snapshot of the store's UI and document state
   */
  getSnapshot() {
    return getSnapshot(this.store);
  }
  /**
   * Loads a snapshot into the editor.
   * @param snapshot - the snapshot to load
   * @returns
   */
  loadSnapshot(snapshot) {
    loadSnapshot(this.store, snapshot);
    return this;
  }
  /**
   * A manager for recording multiple click events.
   *
   * @internal
   */
  _clickManager = new ClickManager(this);
  /**
   * Prevent a double click event from firing the next time the user clicks
   *
   * @public
   */
  cancelDoubleClick() {
    this._clickManager.cancelDoubleClickTimeout();
  }
  /**
   * The previous cursor. Used for restoring the cursor after pan events.
   *
   * @internal
   */
  _prevCursor = "default";
  /** @internal */
  _shiftKeyTimeout = -1;
  /** @internal */
  _setShiftKeyTimeout = () => {
    this.inputs.shiftKey = false;
    this.dispatch({
      type: "keyboard",
      name: "key_up",
      key: "Shift",
      shiftKey: this.inputs.shiftKey,
      ctrlKey: this.inputs.ctrlKey,
      altKey: this.inputs.altKey,
      code: "ShiftLeft"
    });
  };
  /** @internal */
  _altKeyTimeout = -1;
  /** @internal */
  _setAltKeyTimeout = () => {
    this.inputs.altKey = false;
    this.dispatch({
      type: "keyboard",
      name: "key_up",
      key: "Alt",
      shiftKey: this.inputs.shiftKey,
      ctrlKey: this.inputs.ctrlKey,
      altKey: this.inputs.altKey,
      code: "AltLeft"
    });
  };
  /** @internal */
  _ctrlKeyTimeout = -1;
  /** @internal */
  _setCtrlKeyTimeout = () => {
    this.inputs.ctrlKey = false;
    this.dispatch({
      type: "keyboard",
      name: "key_up",
      key: "Ctrl",
      shiftKey: this.inputs.shiftKey,
      ctrlKey: this.inputs.ctrlKey,
      altKey: this.inputs.altKey,
      code: "ControlLeft"
    });
  };
  /** @internal */
  _restoreToolId = "select";
  /** @internal */
  _pinchStart = 1;
  /** @internal */
  _didPinch = false;
  /** @internal */
  _selectedShapeIdsAtPointerDown = [];
  /** @internal */
  _longPressTimeout = -1;
  /** @internal */
  capturedPointerId = null;
  /** @internal */
  performanceTracker;
  /** @internal */
  performanceTrackerTimeout = -1;
  /**
   * Dispatch an event to the editor.
   *
   * @example
   * ```ts
   * editor.dispatch(myPointerEvent)
   * ```
   *
   * @param info - The event info.
   *
   * @public
   */
  dispatch = (info) => {
    this._pendingEventsForNextTick.push(info);
    if (!(info.type === "pointer" && info.name === "pointer_move" || info.type === "wheel" || info.type === "pinch")) {
      this._flushEventsForTick(0);
    }
    return this;
  };
  _pendingEventsForNextTick = [];
  _flushEventsForTick(elapsed) {
    this.run(() => {
      if (this._pendingEventsForNextTick.length > 0) {
        const events = [...this._pendingEventsForNextTick];
        this._pendingEventsForNextTick.length = 0;
        for (const info of events) {
          this._flushEventForTick(info);
        }
      }
      if (elapsed > 0) {
        this.root.handleEvent({ type: "misc", name: "tick", elapsed });
      }
      this.scribbles.tick(elapsed);
    });
  }
  _flushEventForTick = (info) => {
    if (this.getCrashingError()) return this;
    const { inputs } = this;
    const { type } = info;
    if (info.type === "misc") {
      if (info.name === "cancel" || info.name === "complete") {
        this.inputs.isDragging = false;
        if (this.inputs.isPanning) {
          this.inputs.isPanning = false;
          this.setCursor({ type: this._prevCursor, rotation: 0 });
        }
      }
      this.root.handleEvent(info);
      return;
    }
    if (info.shiftKey) {
      clearTimeout(this._shiftKeyTimeout);
      this._shiftKeyTimeout = -1;
      inputs.shiftKey = true;
    } else if (!info.shiftKey && inputs.shiftKey && this._shiftKeyTimeout === -1) {
      this._shiftKeyTimeout = this.timers.setTimeout(this._setShiftKeyTimeout, 150);
    }
    if (info.altKey) {
      clearTimeout(this._altKeyTimeout);
      this._altKeyTimeout = -1;
      inputs.altKey = true;
    } else if (!info.altKey && inputs.altKey && this._altKeyTimeout === -1) {
      this._altKeyTimeout = this.timers.setTimeout(this._setAltKeyTimeout, 150);
    }
    if (info.ctrlKey) {
      clearTimeout(this._ctrlKeyTimeout);
      this._ctrlKeyTimeout = -1;
      inputs.ctrlKey = true;
    } else if (!info.ctrlKey && inputs.ctrlKey && this._ctrlKeyTimeout === -1) {
      this._ctrlKeyTimeout = this.timers.setTimeout(this._setCtrlKeyTimeout, 150);
    }
    const { originPagePoint, currentPagePoint } = inputs;
    if (!inputs.isPointing) {
      inputs.isDragging = false;
    }
    const instanceState = this.store.unsafeGetWithoutCapture(TLINSTANCE_ID);
    const pageState = this.store.get(this._getCurrentPageStateId());
    const cameraOptions = this._cameraOptions.__unsafe__getWithoutCapture();
    switch (type) {
      case "pinch": {
        if (cameraOptions.isLocked) return;
        clearTimeout(this._longPressTimeout);
        this._updateInputsFromEvent(info);
        switch (info.name) {
          case "pinch_start": {
            if (inputs.isPinching) return;
            if (!inputs.isEditing) {
              this._pinchStart = this.getCamera().z;
              if (!this._selectedShapeIdsAtPointerDown.length) {
                this._selectedShapeIdsAtPointerDown = [...pageState.selectedShapeIds];
              }
              this._didPinch = true;
              inputs.isPinching = true;
              this.interrupt();
            }
            return;
          }
          case "pinch": {
            if (!inputs.isPinching) return;
            const {
              point: { z = 1 },
              delta: { x: dx, y: dy }
            } = info;
            const { x, y } = Vec.SubXY(
              info.point,
              instanceState.screenBounds.x,
              instanceState.screenBounds.y
            );
            this.stopCameraAnimation();
            if (instanceState.followingUserId) {
              this.stopFollowingUser();
            }
            const { x: cx, y: cy, z: cz } = unsafe__withoutCapture(() => this.getCamera());
            const { panSpeed, zoomSpeed } = cameraOptions;
            this._setCamera(
              new Vec(
                cx + dx * panSpeed / cz - x / cz + x / (z * zoomSpeed),
                cy + dy * panSpeed / cz - y / cz + y / (z * zoomSpeed),
                z * zoomSpeed
              ),
              { immediate: true }
            );
            return;
          }
          case "pinch_end": {
            if (!inputs.isPinching) return this;
            inputs.isPinching = false;
            const { _selectedShapeIdsAtPointerDown: shapesToReselect } = this;
            this.setSelectedShapes(this._selectedShapeIdsAtPointerDown);
            this._selectedShapeIdsAtPointerDown = [];
            if (this._didPinch) {
              this._didPinch = false;
              if (shapesToReselect.length > 0) {
                this.once("tick", () => {
                  if (!this._didPinch) {
                    this.setSelectedShapes(shapesToReselect);
                  }
                });
              }
            }
            return;
          }
        }
      }
      case "wheel": {
        if (cameraOptions.isLocked) return;
        this._updateInputsFromEvent(info);
        if (this.getIsMenuOpen()) {
        } else {
          const { panSpeed, zoomSpeed, wheelBehavior } = cameraOptions;
          if (wheelBehavior !== "none") {
            this.stopCameraAnimation();
            if (instanceState.followingUserId) {
              this.stopFollowingUser();
            }
            const { x: cx, y: cy, z: cz } = unsafe__withoutCapture(() => this.getCamera());
            const { x: dx, y: dy, z: dz = 0 } = info.delta;
            let behavior = wheelBehavior;
            if (inputs.ctrlKey) behavior = wheelBehavior === "pan" ? "zoom" : "pan";
            switch (behavior) {
              case "zoom": {
                const { x, y } = this.inputs.currentScreenPoint;
                let delta = dz;
                if (wheelBehavior === "zoom") {
                  if (Math.abs(dy) > 10) {
                    delta = 10 * Math.sign(dy) / 100;
                  } else {
                    delta = dy / 100;
                  }
                }
                const zoom = cz + (delta ?? 0) * zoomSpeed * cz;
                this._setCamera(
                  new Vec(
                    cx + (x / zoom - x) - (x / cz - x),
                    cy + (y / zoom - y) - (y / cz - y),
                    zoom
                  ),
                  { immediate: true }
                );
                this.maybeTrackPerformance("Zooming");
                return;
              }
              case "pan": {
                this._setCamera(new Vec(cx + dx * panSpeed / cz, cy + dy * panSpeed / cz, cz), {
                  immediate: true
                });
                this.maybeTrackPerformance("Panning");
                return;
              }
            }
          }
        }
        break;
      }
      case "pointer": {
        if (inputs.isPinching) return;
        this._updateInputsFromEvent(info);
        const { isPen } = info;
        const { isPenMode } = instanceState;
        switch (info.name) {
          case "pointer_down": {
            if (isPenMode && !isPen) return;
            this.clearOpenMenus();
            if (!this.inputs.isPanning) {
              this._longPressTimeout = this.timers.setTimeout(() => {
                this.dispatch({
                  ...info,
                  point: this.inputs.currentScreenPoint,
                  name: "long_press"
                });
              }, this.options.longPressDurationMs);
            }
            this._selectedShapeIdsAtPointerDown = this.getSelectedShapeIds();
            if (info.button === LEFT_MOUSE_BUTTON) this.capturedPointerId = info.pointerId;
            inputs.buttons.add(info.button);
            inputs.isPointing = true;
            inputs.isDragging = false;
            if (!isPenMode && isPen) this.updateInstanceState({ isPenMode: true });
            if (info.button === STYLUS_ERASER_BUTTON) {
              this._restoreToolId = this.getCurrentToolId();
              this.complete();
              this.setCurrentTool("eraser");
            } else if (info.button === MIDDLE_MOUSE_BUTTON) {
              if (!this.inputs.isPanning) {
                this._prevCursor = this.getInstanceState().cursor.type;
              }
              this.inputs.isPanning = true;
              clearTimeout(this._longPressTimeout);
            }
            if (this.inputs.isPanning) {
              this.stopCameraAnimation();
              this.setCursor({ type: "grabbing", rotation: 0 });
              return this;
            }
            break;
          }
          case "pointer_move": {
            if (!isPen && isPenMode) return;
            const { x: cx, y: cy, z: cz } = unsafe__withoutCapture(() => this.getCamera());
            if (this.inputs.isPanning && this.inputs.isPointing) {
              const { currentScreenPoint, previousScreenPoint } = this.inputs;
              const { panSpeed } = cameraOptions;
              const offset = Vec.Sub(currentScreenPoint, previousScreenPoint);
              this.setCamera(
                new Vec(cx + offset.x * panSpeed / cz, cy + offset.y * panSpeed / cz, cz),
                { immediate: true }
              );
              this.maybeTrackPerformance("Panning");
              return;
            }
            if (inputs.isPointing && !inputs.isDragging && Vec.Dist2(originPagePoint, currentPagePoint) * this.getZoomLevel() > (instanceState.isCoarsePointer ? this.options.coarseDragDistanceSquared : this.options.dragDistanceSquared) / cz) {
              inputs.isDragging = true;
              clearTimeout(this._longPressTimeout);
            }
            break;
          }
          case "pointer_up": {
            inputs.isDragging = false;
            inputs.isPointing = false;
            clearTimeout(this._longPressTimeout);
            inputs.buttons.delete(info.button);
            if (this.getIsMenuOpen()) return;
            if (instanceState.isPenMode && !isPen) return;
            if (this.capturedPointerId === info.pointerId) {
              this.capturedPointerId = null;
              info.button = 0;
            }
            if (inputs.isPanning) {
              if (!inputs.keys.has("Space")) {
                inputs.isPanning = false;
              }
              const slideDirection = this.inputs.pointerVelocity;
              const slideSpeed = Math.min(2, slideDirection.len());
              switch (info.button) {
                case LEFT_MOUSE_BUTTON: {
                  this.setCursor({ type: "grab", rotation: 0 });
                  break;
                }
                case MIDDLE_MOUSE_BUTTON: {
                  if (this.inputs.keys.has(" ")) {
                    this.setCursor({ type: "grab", rotation: 0 });
                  } else {
                    this.setCursor({ type: this._prevCursor, rotation: 0 });
                  }
                }
              }
              if (slideSpeed > 0) {
                this.slideCamera({ speed: slideSpeed, direction: slideDirection });
              }
            } else {
              if (info.button === STYLUS_ERASER_BUTTON) {
                this.complete();
                this.setCurrentTool(this._restoreToolId);
              }
            }
            break;
          }
        }
        break;
      }
      case "keyboard": {
        if (info.key === "ShiftRight") info.key = "ShiftLeft";
        if (info.key === "AltRight") info.key = "AltLeft";
        if (info.code === "ControlRight") info.code = "ControlLeft";
        switch (info.name) {
          case "key_down": {
            inputs.keys.add(info.code);
            if (info.code === "Space" && !info.ctrlKey) {
              if (!this.inputs.isPanning) {
                this._prevCursor = instanceState.cursor.type;
              }
              this.inputs.isPanning = true;
              clearTimeout(this._longPressTimeout);
              this.setCursor({ type: this.inputs.isPointing ? "grabbing" : "grab", rotation: 0 });
            }
            break;
          }
          case "key_up": {
            inputs.keys.delete(info.code);
            if (info.code === "Space") {
              if (this.inputs.buttons.has(MIDDLE_MOUSE_BUTTON)) {
              } else {
                this.inputs.isPanning = false;
                this.setCursor({ type: this._prevCursor, rotation: 0 });
              }
            }
            break;
          }
          case "key_repeat": {
            break;
          }
        }
        break;
      }
    }
    if (info.type === "pointer") {
      if (info.button === MIDDLE_MOUSE_BUTTON) {
        info.name = "middle_click";
      } else if (info.button === RIGHT_MOUSE_BUTTON) {
        info.name = "right_click";
      }
      const { isPenMode } = this.store.unsafeGetWithoutCapture(TLINSTANCE_ID);
      if (info.isPen === isPenMode) {
        const clickInfo = this._clickManager.handlePointerEvent(info);
        if (info.name !== clickInfo.name) {
          this.root.handleEvent(info);
          this.emit("event", info);
          this.root.handleEvent(clickInfo);
          this.emit("event", clickInfo);
          return;
        }
      }
    }
    this.root.handleEvent(info);
    this.emit("event", info);
    return this;
  };
  /** @internal */
  maybeTrackPerformance(name) {
    if (debugFlags.measurePerformance.get()) {
      if (this.performanceTracker.isStarted()) {
        clearTimeout(this.performanceTrackerTimeout);
      } else {
        this.performanceTracker.start(name);
      }
      this.performanceTrackerTimeout = this.timers.setTimeout(() => {
        this.performanceTracker.stop();
      }, 50);
    }
  }
}
__decorateClass([
  computed
], Editor.prototype, "getCanUndo", 1);
__decorateClass([
  computed
], Editor.prototype, "getCanRedo", 1);
__decorateClass([
  computed
], Editor.prototype, "getPath", 1);
__decorateClass([
  computed
], Editor.prototype, "getCurrentTool", 1);
__decorateClass([
  computed
], Editor.prototype, "getCurrentToolId", 1);
__decorateClass([
  computed
], Editor.prototype, "getDocumentSettings", 1);
__decorateClass([
  computed
], Editor.prototype, "getInstanceState", 1);
__decorateClass([
  computed
], Editor.prototype, "getOpenMenus", 1);
__decorateClass([
  computed
], Editor.prototype, "getIsMenuOpen", 1);
__decorateClass([
  computed
], Editor.prototype, "getPageStates", 1);
__decorateClass([
  computed
], Editor.prototype, "_getPageStatesQuery", 1);
__decorateClass([
  computed
], Editor.prototype, "getCurrentPageState", 1);
__decorateClass([
  computed
], Editor.prototype, "_getCurrentPageStateId", 1);
__decorateClass([
  computed
], Editor.prototype, "getSelectedShapeIds", 1);
__decorateClass([
  computed
], Editor.prototype, "getSelectedShapes", 1);
__decorateClass([
  computed
], Editor.prototype, "getOnlySelectedShapeId", 1);
__decorateClass([
  computed
], Editor.prototype, "getOnlySelectedShape", 1);
__decorateClass([
  computed
], Editor.prototype, "getSelectionPageBounds", 1);
__decorateClass([
  computed
], Editor.prototype, "getSelectionRotation", 1);
__decorateClass([
  computed
], Editor.prototype, "getSelectionRotatedPageBounds", 1);
__decorateClass([
  computed
], Editor.prototype, "getSelectionRotatedScreenBounds", 1);
__decorateClass([
  computed
], Editor.prototype, "getFocusedGroupId", 1);
__decorateClass([
  computed
], Editor.prototype, "getFocusedGroup", 1);
__decorateClass([
  computed
], Editor.prototype, "getEditingShapeId", 1);
__decorateClass([
  computed
], Editor.prototype, "getEditingShape", 1);
__decorateClass([
  computed
], Editor.prototype, "getHoveredShapeId", 1);
__decorateClass([
  computed
], Editor.prototype, "getHoveredShape", 1);
__decorateClass([
  computed
], Editor.prototype, "getHintingShapeIds", 1);
__decorateClass([
  computed
], Editor.prototype, "getHintingShape", 1);
__decorateClass([
  computed
], Editor.prototype, "getErasingShapeIds", 1);
__decorateClass([
  computed
], Editor.prototype, "getErasingShapes", 1);
__decorateClass([
  computed
], Editor.prototype, "_unsafe_getCameraId", 1);
__decorateClass([
  computed
], Editor.prototype, "getCamera", 1);
__decorateClass([
  computed
], Editor.prototype, "getViewportPageBoundsForFollowing", 1);
__decorateClass([
  computed
], Editor.prototype, "getCameraForFollowing", 1);
__decorateClass([
  computed
], Editor.prototype, "getZoomLevel", 1);
__decorateClass([
  computed
], Editor.prototype, "getViewportScreenBounds", 1);
__decorateClass([
  computed
], Editor.prototype, "getViewportScreenCenter", 1);
__decorateClass([
  computed
], Editor.prototype, "getViewportPageBounds", 1);
__decorateClass([
  computed
], Editor.prototype, "_getCollaboratorsQuery", 1);
__decorateClass([
  computed
], Editor.prototype, "getCollaborators", 1);
__decorateClass([
  computed
], Editor.prototype, "getCollaboratorsOnCurrentPage", 1);
__decorateClass([
  computed
], Editor.prototype, "getRenderingShapes", 1);
__decorateClass([
  computed
], Editor.prototype, "_getAllPagesQuery", 1);
__decorateClass([
  computed
], Editor.prototype, "getPages", 1);
__decorateClass([
  computed
], Editor.prototype, "getCurrentPageId", 1);
__decorateClass([
  computed
], Editor.prototype, "getCurrentPageShapeIdsSorted", 1);
__decorateClass([
  computed
], Editor.prototype, "_getAllAssetsQuery", 1);
__decorateClass([
  computed
], Editor.prototype, "_getShapeGeometryCache", 1);
__decorateClass([
  computed
], Editor.prototype, "_getShapeHandlesCache", 1);
__decorateClass([
  computed
], Editor.prototype, "_getShapePageTransformCache", 1);
__decorateClass([
  computed
], Editor.prototype, "_getShapePageBoundsCache", 1);
__decorateClass([
  computed
], Editor.prototype, "_getShapeClipPathCache", 1);
__decorateClass([
  computed
], Editor.prototype, "_getShapeMaskCache", 1);
__decorateClass([
  computed
], Editor.prototype, "_getShapeMaskedPageBoundsCache", 1);
__decorateClass([
  computed
], Editor.prototype, "_notVisibleShapes", 1);
__decorateClass([
  computed
], Editor.prototype, "getCulledShapes", 1);
__decorateClass([
  computed
], Editor.prototype, "getCurrentPageBounds", 1);
__decorateClass([
  computed
], Editor.prototype, "getCurrentPageShapes", 1);
__decorateClass([
  computed
], Editor.prototype, "getCurrentPageShapesSorted", 1);
__decorateClass([
  computed
], Editor.prototype, "getCurrentPageRenderingShapesSorted", 1);
__decorateClass([
  computed
], Editor.prototype, "_getBindingsIndexCache", 1);
__decorateClass([
  computed
], Editor.prototype, "_getSelectionSharedStyles", 1);
__decorateClass([
  computed({ isEqual: (a, b) => a.equals(b) })
], Editor.prototype, "getSharedStyles", 1);
__decorateClass([
  computed
], Editor.prototype, "getSharedOpacity", 1);
__decorateClass([
  computed
], Editor.prototype, "getIsFocused", 1);
function alertMaxShapes(editor, pageId = editor.getCurrentPageId()) {
  const name = editor.getPage(pageId).name;
  editor.emit("max-shapes", { name, pageId, count: editor.options.maxShapesPerPage });
}
function applyPartialToRecordWithProps(prev, partial) {
  if (!partial) return prev;
  let next = null;
  const entries = Object.entries(partial);
  for (let i = 0, n = entries.length; i < n; i++) {
    const [k, v] = entries[i];
    if (v === void 0) continue;
    if (k === "id" || k === "type" || k === "typeName") continue;
    if (v === prev[k]) continue;
    if (!next) next = { ...prev };
    if (k === "props" || k === "meta") {
      next[k] = { ...prev[k] };
      for (const [nextKey, nextValue] of Object.entries(v)) {
        if (nextValue !== void 0) {
          ;
          next[k][nextKey] = nextValue;
        }
      }
      continue;
    }
    ;
    next[k] = v;
  }
  if (!next) return prev;
  return next;
}
function pushShapeWithDescendants(editor, id, result) {
  const shape = editor.getShape(id);
  if (!shape) return;
  result.push(shape);
  const childIds = editor.getSortedChildIdsForParent(id);
  for (let i = 0, n = childIds.length; i < n; i++) {
    pushShapeWithDescendants(editor, childIds[i], result);
  }
}
function withIsolatedShapes(editor, shapeIds, callback) {
  let result;
  editor.run(
    () => {
      const changes = editor.store.extractingChanges(() => {
        const bindingsWithBoth = /* @__PURE__ */ new Set();
        const bindingsToRemove = /* @__PURE__ */ new Set();
        for (const shapeId of shapeIds) {
          const shape = editor.getShape(shapeId);
          if (!shape) continue;
          for (const binding of editor.getBindingsInvolvingShape(shapeId)) {
            const hasFrom = shapeIds.has(binding.fromId);
            const hasTo = shapeIds.has(binding.toId);
            if (hasFrom && hasTo) {
              bindingsWithBoth.add(binding.id);
              continue;
            }
            if (!hasFrom || !hasTo) {
              bindingsToRemove.add(binding.id);
            }
          }
        }
        editor.deleteBindings([...bindingsToRemove], { isolateShapes: true });
        try {
          result = Result.ok(callback(bindingsWithBoth));
        } catch (error) {
          result = Result.err(error);
        }
      });
      editor.store.applyDiff(reverseRecordsDiff(changes));
    },
    { history: "ignore" }
  );
  if (result.ok) {
    return result.value;
  } else {
    throw result.error;
  }
}
function getCameraFitXFitY(editor, cameraOptions) {
  if (!cameraOptions.constraints) throw Error("Should have constraints here");
  const {
    padding: { x: px, y: py }
  } = cameraOptions.constraints;
  const vsb = editor.getViewportScreenBounds();
  const bounds = Box.From(cameraOptions.constraints.bounds);
  const zx = (vsb.w - px * 2) / bounds.w;
  const zy = (vsb.h - py * 2) / bounds.h;
  return { zx, zy };
}
export {
  Editor
};
//# sourceMappingURL=Editor.mjs.map
